<?php
// /admin/cron/check_links.php
// - BULK (testo): senza parametri
// - AJAX: ?list=1                           -> JSON lista ID
// - AJAX: ?id=123[&fast=1][&debug=1]        -> controlla singolo (JSON; con debug logga header/preview)
require_once __DIR__.'/../../config.php';

$isCli = (PHP_SAPI === 'cli');
if (!$isCli) { require_once __DIR__.'/../../partials/admin_guard.php'; }

ini_set('default_charset','UTF-8');
mb_internal_encoding('UTF-8');
set_time_limit(0);

error_reporting(E_ALL);
ini_set('log_errors', '1');
ini_set('error_log', __DIR__.'/check_links_error.log');

/* ===== Tabella stato ===== */
$pdo->exec("
CREATE TABLE IF NOT EXISTS live_link_status (
  live_id INT PRIMARY KEY,
  url TEXT NOT NULL,
  status ENUM('ok','warn','down') NOT NULL DEFAULT 'warn',
  http_code INT NULL,
  content_type VARCHAR(191) NULL,
  latency_ms INT NULL,
  last_ok DATETIME NULL,
  last_fail DATETIME NULL,
  last_checked DATETIME NOT NULL,
  fail_count INT NOT NULL DEFAULT 0,
  ok_count INT NOT NULL DEFAULT 0,
  last_alert_at DATETIME NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

/* ===== Helpers DB/settings ===== */
function get_setting(PDO $pdo, $k, $def=null){
  try{ $st=$pdo->prepare("SELECT value FROM settings WHERE `key`=?"); $st->execute([$k]); $r=$st->fetch(PDO::FETCH_ASSOC);
       return $r ? $r['value'] : $def; } catch(Throwable $e){ return $def; }
}

/* ===== Helpers contenuto ===== */
function ct_is_m3u8(?string $ct): bool {
  if (!$ct) return false;
  return (bool)preg_match('/(mpegurl|application\/vnd\.apple\.mpegurl|application\/x-mpegurl|audio\/mpegurl|audio\/x-mpegurl|application\/octet-stream)/i', $ct);
}
function body_is_m3u8_like(string $body): bool {
  if ($body === '') return false;

  // UTF-8 diretto o con BOM
  $b = ltrim($body, "\xEF\xBB\xBF");
  // ricerca EXT tipici
  if (stripos($b, '#EXTM3U') !== false) return true;
  if (stripos($b, '#EXTINF') !== false) return true;
  if (preg_match('/#EXT-X-[A-Z0-9\-]+/i', $b)) return true;

  // UTF-16 LE/BE pattern di "#EXTM3U"
  if (strpos($body, "\x23\x00\x45\x00\x58\x00\x54\x00\x4D\x00\x33\x00\x55\x00") !== false) return true; // LE
  if (strpos($body, "\x00\x23\x00\x45\x00\x58\x00\x54\x00\x4D\x00\x33\x00\x55\x00") !== false) return true; // BE

  // tenta conversione charset
  $enc = @mb_detect_encoding($body, ['UTF-8','UTF-16LE','UTF-16BE','ISO-8859-1','Windows-1252'], true);
  if ($enc && $enc !== 'UTF-8') {
    $txt = @mb_convert_encoding($body, 'UTF-8', $enc);
    if ($txt !== false) {
      if (stripos($txt, '#EXTM3U') !== false) return true;
      if (stripos($txt, '#EXTINF') !== false) return true;
      if (preg_match('/#EXT-X-[A-Z0-9\-]+/i', $txt)) return true;
    }
  }
  return false;
}
function body_preview_str(string $body, int $max=200): string {
  $s = substr($body, 0, $max);
  $s = preg_replace('/[^\P{C}\t\r\n]/u', '?', $s); // rimuove non-printable
  return $s;
}

/* ===== Core curl ===== */
function curl_fetch(string $url, array $opt = []): array {
  $headers = $opt['headers'] ?? [];
  $ua      = $opt['ua'] ?? 'LinkMonitor/1.4';
  $timeout = $opt['timeout'] ?? 12;
  $connect = $opt['connect_timeout'] ?? 6;
  $maxredir= $opt['max_redirs'] ?? 8;
  $httpver = $opt['http_version'] ?? CURL_HTTP_VERSION_2TLS;
  $range   = $opt['range'] ?? null;
  $referer = $opt['referer'] ?? null;
  $cookie  = $opt['cookie'] ?? null;
  $cookieFile = $opt['cookie_file'] ?? null;
  $ipresolve  = $opt['ipresolve'] ?? null;

  $baseHeaders = [
    'Accept: application/x-mpegURL, application/vnd.apple.mpegurl, */*',
    'Accept-Language: it-IT,it;q=0.9,en-US;q=0.8,en;q=0.7',
    'Cache-Control: no-cache',
    'Pragma: no-cache',
    'Connection: keep-alive'
  ];
  if ($range)   $baseHeaders[] = 'Range: bytes='.$range;
  if ($referer) { $baseHeaders[] = 'Referer: '.$referer; $baseHeaders[] = 'Origin: '.rtrim($referer,'/'); }
  $headers = array_merge($baseHeaders, $headers);

  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_MAXREDIRS      => $maxredir,
    CURLOPT_CONNECTTIMEOUT => $connect,
    CURLOPT_TIMEOUT        => $timeout,
    CURLOPT_HTTPHEADER     => $headers,
    CURLOPT_USERAGENT      => $ua,
    CURLOPT_SSL_VERIFYPEER => false,
    CURLOPT_SSL_VERIFYHOST => 0,
    CURLOPT_ENCODING       => '', // gzip/br
    CURLOPT_HTTP_VERSION   => $httpver,
    CURLOPT_HEADER         => false
  ]);
  if ($ipresolve !== null && defined('CURL_IPRESOLVE_V4')) {
    @curl_setopt($ch, CURLOPT_IPRESOLVE, $ipresolve);
  }
  if ($cookieFile) {
    curl_setopt($ch, CURLOPT_COOKIEFILE, $cookieFile);
    curl_setopt($ch, CURLOPT_COOKIEJAR,  $cookieFile);
  }
  if ($cookie) {
    curl_setopt($ch, CURLOPT_COOKIE, $cookie);
  }

  $t0   = microtime(true);
  $body = curl_exec($ch);
  $err  = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  $ct   = (string)curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
  $sz   = (int)curl_getinfo($ch, CURLINFO_SIZE_DOWNLOAD);
  curl_close($ch);
  $ms = (int)round((microtime(true)-$t0)*1000);

  return [
    'ok'   => ($body !== false && $code >= 200 && $code <= 206),
    'body' => $body === false ? '' : $body,
    'code' => $code,
    'ct'   => $ct,
    'size' => $sz,
    'ms'   => $ms,
    'err'  => $body === false ? ($err ?: 'curl') : null,
  ];
}

/* ===== Referer per-live ===== */
function compute_referer(PDO $pdo, int $liveId): ?string {
  // 1) se esiste un template in settings (es. https://supersport.xo.je/client/live.php?id={id})
  $tpl = get_setting($pdo, 'm3u8_referer', null);
  if ($tpl && strpos($tpl, '{id}') !== false) {
    return str_replace('{id}', (string)$liveId, $tpl);
  }
  // 2) se chiamato via web, usa il dominio corrente con la pagina player
  if (!empty($_SERVER['HTTP_HOST'])) {
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    return $scheme.'://'.$_SERVER['HTTP_HOST'].'/client/live.php?id='.$liveId;
  }
  // 3) CLI senza template -> nessun referer (alcuni CDN lo richiederanno comunque)
  return null;
}

/**
 * Check m3u8 robusto con Referer per-live.
 */
function check_m3u8(string $url, int $liveId, bool $fast=false, bool $debug=false): array {
  $url = trim($url);

  $ua_probe   = 'LinkMonitor/1.4';
  $ua_browser = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/124.0 Safari/537.36';

  $connectTimeout = $fast ? 3 : 6;
  $timeout        = $fast ? 6 : 12;
  $maxRedirs      = $fast ? 5 : 8;
  $rangeBytes     = $fast ? '0-8191' : '0-16383';

  $referer = compute_referer($GLOBALS['pdo'], $liveId);
  $forwardCookie = $_SERVER['HTTP_COOKIE'] ?? null; // se chiamato via admin/ajax
  $cookieFile = sys_get_temp_dir().'/m3u8_cj_'.sha1($url.($forwardCookie ?? '').microtime(true)).'.txt';

  // opzionale: preflight sulla referer per cookie server-side
  if ($referer) {
    curl_fetch($referer, [
      'ua'=>$ua_browser, 'timeout'=>$timeout, 'connect_timeout'=>$connectTimeout,
      'max_redirs'=>$maxRedirs, 'cookie_file'=>$cookieFile, 'cookie'=>$forwardCookie
    ]);
  }

  $attempts = [];

  // 1) HTTP/2 + UA tecnico + Range
  $attempts[] = curl_fetch($url, [
    'ua'=>$ua_probe, 'timeout'=>$timeout, 'connect_timeout'=>$connectTimeout, 'max_redirs'=>$maxRedirs,
    'http_version'=>CURL_HTTP_VERSION_2TLS, 'range'=>$rangeBytes, 'referer'=>$referer,
    'cookie_file'=>$cookieFile, 'cookie'=>$forwardCookie
  ]);

  // 2) HTTP/1.1 + UA browser + NO Range
  $attempts[] = curl_fetch($url, [
    'ua'=>$ua_browser, 'timeout'=>$timeout, 'connect_timeout'=>$connectTimeout, 'max_redirs'=>$maxRedirs,
    'http_version'=>CURL_HTTP_VERSION_1_1, 'referer'=>$referer,
    'cookie_file'=>$cookieFile, 'cookie'=>$forwardCookie
  ]);

  // 3) HTTP/1.1 + UA browser + Range
  $attempts[] = curl_fetch($url, [
    'ua'=>$ua_browser, 'timeout'=>$timeout, 'connect_timeout'=>$connectTimeout, 'max_redirs'=>$maxRedirs,
    'http_version'=>CURL_HTTP_VERSION_1_1, 'range'=>$rangeBytes, 'referer'=>$referer,
    'cookie_file'=>$cookieFile, 'cookie'=>$forwardCookie
  ]);

  // 4) forzatura IPv4
  $attempts[] = curl_fetch($url, [
    'ua'=>$ua_browser, 'timeout'=>$timeout, 'connect_timeout'=>$connectTimeout, 'max_redirs'=>$maxRedirs,
    'http_version'=>CURL_HTTP_VERSION_1_1, 'referer'=>$referer,
    'cookie_file'=>$cookieFile, 'cookie'=>$forwardCookie,
    'ipresolve'=>defined('CURL_IPRESOLVE_V4') ? CURL_IPRESOLVE_V4 : null
  ]);

  $reasons = [];
  $lastWarn = null;

  foreach ($attempts as $a) {
    $reasons[] = $a['code'].'/'.($a['ct'] ?: '-').'/'.($a['err'] ?: 'ok').'/size='.$a['size'];
    if (!$a['ok']) continue;

    if (body_is_m3u8_like($a['body'])) {
      if ($debug) {
        @file_put_contents(__DIR__.'/check_links_debug.log',
          "[OK] code={$a['code']} ct={$a['ct']} size={$a['size']} body=".body_preview_str($a['body'])."\n",
          FILE_APPEND);
      }
      return ['status'=>'ok','code'=>$a['code'],'ct'=>$a['ct'],'ms'=>$a['ms'],'reason'=>'m3u8'];
    }

    // se Content-Type è "mpegurl" e c'è corpo (>0), accetta comunque OK
    if ($a['size'] > 0 && ct_is_m3u8($a['ct'])) {
      if ($debug) {
        @file_put_contents(__DIR__.'/check_links_debug.log',
          "[OK-CT] code={$a['code']} ct={$a['ct']} size={$a['size']} body=".body_preview_str($a['body'])."\n",
          FILE_APPEND);
      }
      return ['status'=>'ok','code'=>$a['code'],'ct'=>$a['ct'],'ms'=>$a['ms'],'reason'=>'ct-mpegurl'];
    }

    // 200/206 ma non riconoscibile -> warn
    $lastWarn = ['status'=>'warn','code'=>$a['code'],'ct'=>$a['ct'],'ms'=>$a['ms'],'reason'=>'no-ext-tags'];
  }

  // blocchi classici
  foreach ($attempts as $a) {
    if (in_array($a['code'], [401,403,451], true)) {
      return ['status'=>'down','code'=>$a['code'],'ct'=>$a['ct'],'ms'=>$a['ms'],'reason'=>'blocked-'.$a['code']];
    }
  }

  if ($lastWarn) return $lastWarn;

  $w = end($attempts);
  $why = $w['err'] ? $w['err'] : ('http-'.$w['code']);
  return ['status'=>'down','code'=>$w['code'],'ct'=>$w['ct'],'ms'=>$w['ms'],'reason'=>$why.' | tried='.implode(' || ', $reasons)];
}

function upsert_status(PDO $pdo, int $id, string $url, array $res): void {
  $now = date('Y-m-d H:i:s');
  $st = $pdo->prepare("SELECT 1 FROM live_link_status WHERE live_id=?");
  $st->execute([$id]);
  $exists = (bool)$st->fetchColumn();

  if ($exists) {
    $sql = "UPDATE live_link_status
            SET url=?, status=?, http_code=?, content_type=?, latency_ms=?, last_checked=?,
                last_ok   = IF(?='ok',   ?, last_ok),
                last_fail = IF(?='down', ?, last_fail),
                ok_count  = ok_count  + (?='ok'),
                fail_count= fail_count+ (?='down')
            WHERE live_id=?";
    $pdo->prepare($sql)->execute([
      $url, $res['status'], $res['code'], $res['ct'], $res['ms'], $now,
      $res['status'], $now,
      $res['status'], $now,
      $res['status'], $res['status'],
      $id
    ]);
  } else {
    $pdo->prepare("INSERT INTO live_link_status
      (live_id,url,status,http_code,content_type,latency_ms,last_ok,last_fail,last_checked,ok_count,fail_count)
      VALUES (?,?,?,?,?,?,?,?,?, ?, ?)")
      ->execute([
        $id, $url, $res['status'], $res['code'], $res['ct'], $res['ms'],
        $res['status']==='ok'  ? $now : null,
        $res['status']==='down'? $now : null,
        $now,
        $res['status']==='ok'  ? 1 : 0,
        $res['status']==='down'? 1 : 0
      ]);
  }
}

/* ====== AJAX: lista ====== */
if (isset($_GET['list'])) {
  header('Content-Type: application/json; charset=UTF-8');
  $st = $pdo->query("SELECT id,title,m3u8_url FROM lives
                     WHERE is_active=1 AND m3u8_url IS NOT NULL AND m3u8_url<>''
                     ORDER BY id ASC");
  $items = [];
  foreach ($st->fetchAll(PDO::FETCH_ASSOC) as $r) {
    $items[] = ['id'=>(int)$r['id'], 'title'=>$r['title'], 'url'=>$r['m3u8_url']];
  }
  echo json_encode(['total'=>count($items),'items'=>$items], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
  exit;
}

/* ====== AJAX: singolo ID ====== */
if (isset($_GET['id'])) {
  header('Content-Type: application/json; charset=UTF-8');
  $id = (int)$_GET['id'];
  $fast  = !empty($_GET['fast']);
  $debug = !empty($_GET['debug']);
  $st = $pdo->prepare("SELECT id,title,m3u8_url FROM lives WHERE id=? AND m3u8_url IS NOT NULL AND m3u8_url<>''");
  $st->execute([$id]);
  $row = $st->fetch(PDO::FETCH_ASSOC);
  if (!$row) { http_response_code(404); echo json_encode(['error'=>'not-found']); exit; }

  $res = check_m3u8($row['m3u8_url'], (int)$row['id'], $fast, $debug);
  upsert_status($pdo, (int)$row['id'], $row['m3u8_url'], $res);

  echo json_encode([
    'id'=>(int)$row['id'],
    'title'=>$row['title'],
    'status'=>$res['status'],
    'http_code'=>$res['code'],
    'content_type'=>$res['ct'],
    'latency_ms'=>$res['ms'],
    'reason'=>$res['reason'] ?? null
  ], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
  exit;
}

/* ====== BULK (manuale/CLI; output testuale) ====== */
if (!$isCli) header('Content-Type: text/plain; charset=UTF-8');

$st = $pdo->query("SELECT id,title,m3u8_url FROM lives
                   WHERE is_active=1 AND m3u8_url IS NOT NULL AND m3u8_url<>''");
$lives = $st->fetchAll(PDO::FETCH_ASSOC);
foreach ($lives as $row) {
  $res = check_m3u8($row['m3u8_url'], (int)$row['id'], false, false);
  upsert_status($pdo, (int)$row['id'], $row['m3u8_url'], $res);
  if (!$isCli) {
    echo sprintf("#%d %s — %s (%d) %dms — %s\n",
      $row['id'], $row['title'], strtoupper($res['status']), $res['code'], $res['ms'], $res['reason'] ?? '-');
    @ob_flush(); @flush();
  }
}
if (!$isCli) echo "Done.\n";
