<?php
// /admin/lives.php
$page_title = 'Live';
require_once __DIR__ . '/../partials/admin_guard.php';

if (!headers_sent()) header('Content-Type: text/html; charset=UTF-8');
ini_set('default_charset','UTF-8');
mb_internal_encoding('UTF-8');

/* ---------- Utils ---------- */
function get_setting($pdo,$k){
  $st=$pdo->prepare("SELECT value FROM settings WHERE `key`=?");
  $st->execute([$k]); $r=$st->fetch();
  return $r ? $r['value'] : null;
}
function set_setting($pdo,$k,$v){
  $st=$pdo->prepare("INSERT INTO settings(`key`,`value`) VALUES(?,?)
                     ON DUPLICATE KEY UPDATE value=VALUES(value)");
  $st->execute([$k,$v]);
}
function slugify($s){ $s=strtolower(preg_replace('/[^a-z0-9]+/i','-',$s)); return trim($s,'-'); }
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function markQ($s,$q){ if($q==='') return h($s); $safe=h($s); $sq=h($q); return str_ireplace($sq,'<mark>'.$sq.'</mark>',$safe); }
function dt_to_mysql($str){
  $str=trim((string)$str); if($str==='') return null;
  $ts=strtotime($str); if($ts===false) $ts=strtotime(str_replace('T',' ',$str));
  return $ts ? date('Y-m-d H:i:s',$ts) : null;
}
function dt_to_input($mysql){
  if(!$mysql) return ''; $ts=strtotime($mysql);
  return $ts ? date('Y-m-d\TH:i',$ts) : '';
}
function safe_unlink(?string $path): void{
  if(!$path) return;
  if(strpos($path, '/uploads/')!==0) return;
  $full=rtrim($_SERVER['DOCUMENT_ROOT']??'','/').$path;
  if(is_file($full) && is_writable($full)) @unlink($full);
}
function handle_m3u8_upload(string $field='m3u8_file'): ?string{
  if(empty($_FILES[$field]['name']) || !is_uploaded_file($_FILES[$field]['tmp_name'])) return null;
  $name = $_FILES[$field]['name'];
  $tmp  = $_FILES[$field]['tmp_name'];
  $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
  if($ext!=='m3u8') $ext='m3u8';
  if(($_FILES[$field]['size'] ?? 0) > 5*1024*1024) return null;
  @mkdir($_SERVER['DOCUMENT_ROOT'].'/uploads/m3u8',0775,true);
  try{ $rand = bin2hex(random_bytes(4)); }catch(Throwable $e){ $rand = dechex(mt_rand()); }
  $fname = '/uploads/m3u8/live_'.time().'_'.$rand.'.'.$ext;
  if(move_uploaded_file($tmp, $_SERVER['DOCUMENT_ROOT'].$fname)) return $fname;
  return null;
}

/* Parse/normalizza embed multipli */
function parse_embed_variants($val): array {
  $val = trim((string)$val);
  if ($val === '') return [];
  if ($val[0] === '[' || $val[0] === '{') {
    $j = json_decode($val, true);
    if (json_last_error() === JSON_ERROR_NONE) {
      if (isset($j['list']) && is_array($j['list'])) $j = $j['list'];
      $out=[];
      foreach ($j as $i=>$it) {
        if (is_string($it)) {
          $out[] = ['label'=>'Embed '.($i+1), 'code'=>$it, 'sort'=>0];
        } elseif (is_array($it) && !empty($it['code'])) {
          $out[] = [
            'label'=>trim((string)($it['label'] ?? ('Embed '.($i+1)))),
            'code' =>trim((string)$it['code']),
            'sort' =>(int)($it['sort'] ?? 0),
          ];
        }
      }
      return $out;
    }
  }
  return [['label'=>'Embed 1','code'=>$val,'sort'=>0]];
}

/* ====== Auto-cover per competizione ====== */
function detect_competition_image(string $title, string $note=''): ?string {
  $t = mb_strtolower($title.' '.$note, 'UTF-8');

  $rules = [
    // campionati italiani
    ['pattern'=>['serie a','zona serie a','diretta goal serie a'], 'file'=>'serie-a.png'],
    ['pattern'=>['serie b'],                                        'file'=>'serie-b.png'],
    ['pattern'=>['serie c'],                                        'file'=>'serie-c.png'],
    // coppe europee
    ['pattern'=>['champions league','ucl'],                         'file'=>'champions-league.png'],
    ['pattern'=>['europa league','europe league','uel'],            'file'=>'europa-league.png'],
    ['pattern'=>['conference league','uecl'],                       'file'=>'conference-league.png'],
    // motori
    ['pattern'=>['motogp'],                                         'file'=>'motogp.png'],
    ['pattern'=>['f1','formula 1','formula one'],                   'file'=>'f1.png'],
  ];

  foreach ($rules as $r) {
    foreach ($r['pattern'] as $kw) {
      if (strpos($t, $kw) !== false) {
        $base = '/uploads/competitions/';
        if ($r['file']==='europa-league.png') {
          $p1 = ($_SERVER['DOCUMENT_ROOT'] ?? '').$base.'europa-league.png';
          $p2 = ($_SERVER['DOCUMENT_ROOT'] ?? '').$base.'europe-league.png';
          if (is_file($p1)) return $base.'europa-league.png';
          if (is_file($p2)) return $base.'europe-league.png';
          return $base.'europa-league.png';
        }
        return $base.$r['file'];
      }
    }
  }
  return null;
}

/* ====== Assicura la colonna linked_live_id ====== */
function ensure_linked_column(PDO $pdo): void {
  static $done=false; if($done) return; $done=true;
  try{
    $db = $pdo->query("SELECT DATABASE()")->fetchColumn();
    $chk=$pdo->prepare("SELECT COUNT(*) FROM information_schema.COLUMNS
                        WHERE TABLE_SCHEMA=? AND TABLE_NAME='lives' AND COLUMN_NAME='linked_live_id'");
    $chk->execute([$db]);
    if(!$chk->fetchColumn()){
      $pdo->exec("ALTER TABLE lives ADD COLUMN linked_live_id INT NULL DEFAULT NULL");
      try{
        $pdo->exec("ALTER TABLE lives ADD CONSTRAINT fk_lives_linked_live
                    FOREIGN KEY (linked_live_id) REFERENCES lives(id)
                    ON DELETE SET NULL ON UPDATE CASCADE");
      }catch(Throwable $e){ /* ok anche senza FK */ }
    }
  }catch(Throwable $e){ /* ignora */ }
}
ensure_linked_column($pdo);

/* ---------- AJAX: ricerca live per collegamento ---------- */
if (isset($_GET['ajax']) && $_GET['ajax'] === 'find_live') {
  header('Content-Type: application/json; charset=UTF-8');
  $q = trim($_GET['q'] ?? '');
  $lim = 20;
  if ($q === '') { echo json_encode([]); exit; }
  $params = [];
  $where  = ["1=1"];
  if (ctype_digit($q)) {
    $where[] = "(l.id = ? OR l.title LIKE ? OR l.slug LIKE ?)";
    $params[] = (int)$q; $params[] = "%$q%"; $params[] = "%$q%";
  } else {
    $where[] = "(l.title LIKE ? OR l.slug LIKE ?)";
    $params[] = "%$q%"; $params[] = "%$q%";
  }
  $sql = "SELECT l.id, l.title, l.slug, c.name AS cat
          FROM lives l
          JOIN categories c ON c.id=l.category_id
          WHERE ".implode(' AND ',$where)."
          ORDER BY l.title ASC
          LIMIT $lim";
  $st = $pdo->prepare($sql); $st->execute($params);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC);
  echo json_encode($rows, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
  exit;
}

/* ---------- URL corrente (PRG) + variante SENZA 'edit' ---------- */
$qs   = $_SERVER['QUERY_STRING'] ?? '';
$self = '/admin/lives.php'.($qs ? '?'.$qs : '');
$qsArr = $_GET; unset($qsArr['edit']);
$qsNoEdit     = http_build_query($qsArr);
$self_no_edit = '/admin/lives.php'.($qsNoEdit ? '?'.$qsNoEdit : '');

/* ===== Import CSV Lives (categoria fissa + auto-cover) ===== */
if (($_GET['action'] ?? '') === 'import_csv' && $_SERVER['REQUEST_METHOD']==='POST') {

  // parser data/ora in italiano
  function parse_it_datetime($data_it, $ora_it): ?string {
    $data_it = trim((string)$data_it);
    $ora_it  = trim((string)$ora_it);
    if ($data_it === '') return null;
    $Y=$m=$d=null;

    if (preg_match('~^(\d{1,2})/(\d{1,2})/(\d{4})$~', $data_it, $mm)) {
      $d=(int)$mm[1]; $m=(int)$mm[2]; $Y=(int)$mm[3];
    } elseif (preg_match('~^(\d{1,2})\s+([A-Za-zÀ-ÿ]+)\s+(\d{4})$~u', $data_it, $mm)) {
      $d=(int)$mm[1]; $nome=strtolower($mm[2]); $Y=(int)$mm[3];
      $map = [
        'gennaio'=>1,'febbraio'=>2,'marzo'=>3,'aprile'=>4,'maggio'=>5,'giugno'=>6,
        'luglio'=>7,'agosto'=>8,'settembre'=>9,'ottobre'=>10,'novembre'=>11,'dicembre'=>12
      ];
      $m = $map[$nome] ?? null;
    }
    if (!$Y || !$m || !$d) return null;

    $H=0; $i=0;
    if ($ora_it !== '' && preg_match('~^(\d{1,2}):(\d{2})$~',$ora_it,$tt)) {
      $H=(int)$tt[1]; $i=(int)$tt[2];
    }
    return sprintf('%04d-%02d-%02d %02d:%02d:00',$Y,$m,$d,$H,$i);
  }

  function get_or_create_category_id(PDO $pdo, string $name): int {
    $name = trim($name);
    if ($name==='') $name='Generale';
    $st = $pdo->prepare("SELECT id FROM categories WHERE name=? LIMIT 1");
    $st->execute([$name]);
    $id = $st->fetchColumn();
    if ($id) return (int)$id;
    $slug = slugify($name);
    $ins  = $pdo->prepare("INSERT INTO categories(name, slug, created_at) VALUES(?,?,NOW())");
    $ins->execute([$name,$slug]);
    return (int)$pdo->lastInsertId();
  }

  function unique_slug(PDO $pdo, string $base): string {
    $slug = slugify($base);
    if ($slug==='') $slug = 'live';
    $st = $pdo->prepare("SELECT COUNT(*) FROM lives WHERE slug=?");
    $try = $slug; $i=2;
    while (true) {
      $st->execute([$try]);
      if ((int)$st->fetchColumn()===0) return $try;
      $try = $slug.'-'.$i++;
    }
  }

  if (!isset($_FILES['csv']) || $_FILES['csv']['error']!==UPLOAD_ERR_OK) {
    header('Location: '.$self_no_edit.'&import_msg='.urlencode('Caricamento file fallito')); exit;
  }

  $fp = fopen($_FILES['csv']['tmp_name'], 'r');
  if (!$fp) { header('Location: '.$self_no_edit.'&import_msg='.urlencode('Impossibile leggere il CSV')); exit; }

  // mappa intestazioni
  $hdr = fgetcsv($fp, 0, ',');
  if (!$hdr){ header('Location: '.$self_no_edit.'&import_msg='.urlencode('CSV senza intestazione')); exit; }
  $map = [];
  foreach ($hdr as $i=>$c) $map[strtolower(trim($c))]=$i;

  // colonne senza "categoria": verrà forzata a "Programmazione Eventi"
  $need = ['data_it','ora_it','titolo','note','is_active','always_live','slug_custom','thumb_url','fonte','label_fornitore'];
  foreach ($need as $c) if(!isset($map[$c])){
    header('Location: '.$self_no_edit.'&import_msg='.urlencode("Colonna mancante: $c")); exit;
  }

  $ins = $pdo->prepare("INSERT INTO lives
      (category_id,title,slug,m3u8_url,embed_code,image_path,description,event_at,always_live,is_active,sort_order,linked_live_id)
      VALUES (?,?,?,?,?,?,?,?,?,?,?,NULL)");

  // categoria fissa
  $cat_id_fixed = get_or_create_category_id($pdo, 'Programmazione Eventi');

  $ok=0; $skipped=0; $rown=1;
  while (($r = fgetcsv($fp, 0, ',')) !== false) {
    $rown++;

    $data_it   = $r[$map['data_it']]   ?? '';
    $ora_it    = $r[$map['ora_it']]    ?? '';
    $titolo    = trim($r[$map['titolo']] ?? '');
    $note      = trim($r[$map['note']] ?? '');
    $is_active = trim((string)($r[$map['is_active']] ?? '1')) !== '0' ? 1 : 0;
    $always    = trim((string)($r[$map['always_live']] ?? '0'))==='1' ? 1 : 0;
    $slug_cus  = trim($r[$map['slug_custom']] ?? '');
    $thumb_url = trim($r[$map['thumb_url']] ?? '');
    $fonte     = trim($r[$map['fonte']] ?? '');
    $fornitore = trim($r[$map['label_fornitore']] ?? '');

    if ($titolo===''){ $skipped++; continue; }

    $event_at = $always ? null : parse_it_datetime($data_it,$ora_it);

    // descrizione: note + eventuali info
    $descr = $note;
    if ($fornitore!=='') $descr .= ($descr? ' • ': '')."Contenuto fornito da: ".$fornitore;
    if ($fonte!=='')     $descr .= ($descr? ' • ': '')."Fonte: ".$fonte;

    // slug
    $slug = $slug_cus !== '' ? slugify($slug_cus) : unique_slug($pdo, $titolo);

    // auto-cover se non fornita
    if ($thumb_url === '') {
      $auto = detect_competition_image($titolo, $note);
      if ($auto) $thumb_url = $auto;
    }

    try{
      $ins->execute([
        $cat_id_fixed,    // categoria fissa
        $titolo,
        $slug,
        '',               // m3u8_url
        '',               // embed_code
        $thumb_url,       // image_path (può essere path locale o URL)
        $descr,
        $event_at,
        $always,
        $is_active,
        0                 // sort_order
      ]);
      $ok++;
    }catch(Throwable $e){
      $skipped++;
    }
  }
  fclose($fp);

  $msg = "Import completato: $ok inserite, $skipped saltate.";
  header('Location: '.$self_no_edit.'&import_ok='.$ok.'&import_skip='.$skipped.'&import_msg='.urlencode($msg));
  exit;
}

/* ---------- POST handlers ---------- */
if($_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['gen_playlist_token'])){
  try{ $new=bin2hex(random_bytes(24)); }catch(Exception $e){ $new=bin2hex(openssl_random_pseudo_bytes(24)); }
  set_setting($pdo,'playlist_token',$new);
  header('Location: '.$self_no_edit); exit;
}
if($_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['clear_playlist_token'])){
  set_setting($pdo,'playlist_token','');
  header('Location: '.$self_no_edit); exit;
}

/* Set attivo/disattivo per una singola live */
if ($_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['set_active'])) {
  $id  = (int)$_POST['id'];
  $val = (int)((int)($_POST['active'] ?? 0) ? 1 : 0);
  if ($id > 0) $pdo->prepare("UPDATE lives SET is_active=? WHERE id=?")->execute([$val, $id]);
  header('Location: '.$self_no_edit); exit;
}

/* === BULK ACTIONS === */
if($_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['bulk_action'])){
  $act = trim((string)($_POST['bulk_action'] ?? ''));
  $ids = [];
  if(isset($_POST['ids']) && is_array($_POST['ids'])){
    foreach($_POST['ids'] as $id){ $id=(int)$id; if($id>0) $ids[]=$id; }
  }
  $ids = array_values(array_unique($ids));
  if(!$ids){ header('Location: '.$self_no_edit); exit; }
  $in  = implode(',', array_fill(0, count($ids), '?'));

  if ($act === 'activate' || $act === 'deactivate') {
    $val = ($act === 'activate') ? 1 : 0;
    $st  = $pdo->prepare("UPDATE lives SET is_active = ? WHERE id IN ($in)");
    $st->execute(array_merge([$val], $ids));
  } elseif ($act === 'delete') {
    $st = $pdo->prepare("SELECT id,image_path,m3u8_url FROM lives WHERE id IN ($in)");
    $st->execute($ids);
    foreach($st->fetchAll() as $r){
      if(!empty($r['image_path'])) safe_unlink($r['image_path']);
      if(!empty($r['m3u8_url']) && strpos($r['m3u8_url'],'/uploads/')===0) safe_unlink($r['m3u8_url']);
    }
    $pdo->prepare("DELETE FROM lives WHERE id IN ($in)")->execute($ids);
  }
  header('Location: '.$self_no_edit); exit;
}

/* CRUD singole */
if($_SERVER['REQUEST_METHOD']==='POST'){
  if(isset($_POST['create'])){
    $title=trim($_POST['title']); $slug=slugify($title);
    $cat=(int)$_POST['category_id'];
    $m3u8=trim($_POST['m3u8_url'] ?? '');
    $linked_live_id = (int)$_POST['linked_live_id'] ?: null;

    $embed_single = trim($_POST['embed_code'] ?? '');
    $labels = $_POST['embeds']['label'] ?? [];
    $codes  = $_POST['embeds']['code']  ?? [];
    $orders = $_POST['embeds']['sort']  ?? [];
    $multi=[];
    for ($i=0; $i < max(count($labels), count($codes)); $i++){
      $code = trim($codes[$i] ?? ''); if ($code==='') continue;
      $multi[] = ['label'=>trim($labels[$i] ?? '') ?: ('Embed '.(count($multi)+1)), 'code'=>$code, 'sort'=>(int)($orders[$i] ?? 0)];
    }
    $used_multi = !empty($_POST['embeds_present']);
    $embed = $used_multi ? ($multi ? json_encode($multi, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE) : '') : $embed_single;

    $sort=(int)($_POST['sort_order']??0);
    $desc=trim($_POST['description']??'');
    $always_live=empty($_POST['always_live'])?0:1;
    $event_at=$always_live?null:dt_to_mysql($_POST['event_at']??'');

    $upM3u8 = handle_m3u8_upload('m3u8_file'); if($upM3u8) $m3u8=$upM3u8;

    $imgPath=null;
    if(!empty($_FILES['image']['name']) && is_uploaded_file($_FILES['image']['tmp_name'])){
      @mkdir($_SERVER['DOCUMENT_ROOT'].'/uploads',0775,true);
      $ext=strtolower(pathinfo($_FILES['image']['name'],PATHINFO_EXTENSION)?:'jpg');
      $ext=preg_replace('/[^a-z0-9]/','',$ext)?:'jpg';
      try{ $rand = bin2hex(random_bytes(3)); }catch(Throwable $e){ $rand = dechex(mt_rand()); }
      $fname='/uploads/live_'.time().'_'.$rand.'.'.$ext;
      if(move_uploaded_file($_FILES['image']['tmp_name'], $_SERVER['DOCUMENT_ROOT'].$fname)) $imgPath=$fname;
    } else {
      // se non carichi immagine, prova auto-cover
      $imgPath = detect_competition_image($title, $desc) ?: null;
    }

    $stmt=$pdo->prepare("INSERT INTO lives(category_id,title,slug,m3u8_url,embed_code,image_path,description,event_at,always_live,is_active,sort_order,linked_live_id)
                         VALUES(?,?,?,?,?,?,?,?,?,?,?,?)");
    $stmt->execute([$cat,$title,$slug,$m3u8,$embed,$imgPath,$desc,$event_at,$always_live,1,$sort,$linked_live_id]);
    header('Location: '.$self_no_edit); exit;
  }

  if(isset($_POST['update'])){
    $id=(int)$_POST['id'];
    $title=trim($_POST['title']); $slug=slugify($title);
    $cat=(int)$_POST['category_id'];
    $m3u8=trim($_POST['m3u8_url'] ?? '');
    $linked_live_id = (int)($_POST['linked_live_id'] ?? 0);
    if ($linked_live_id<=0 || $linked_live_id===$id) $linked_live_id=null;

    $embed_single = trim($_POST['embed_code'] ?? '');
    $labels = $_POST['embeds']['label'] ?? [];
    $codes  = $_POST['embeds']['code']  ?? [];
    $orders = $_POST['embeds']['sort']  ?? [];
    $multi=[];
    for ($i=0; $i < max(count($labels), count($codes)); $i++){
      $code = trim($codes[$i] ?? ''); if ($code==='') continue;
      $multi[] = ['label'=>trim($labels[$i] ?? '') ?: ('Embed '.(count($multi)+1)), 'code'=>$code, 'sort'=>(int)($orders[$i] ?? 0)];
    }
    $used_multi = !empty($_POST['embeds_present']);
    $embed = $used_multi ? ($multi ? json_encode($multi, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE) : '') : $embed_single;

    $sort=(int)($_POST['sort_order']??0);
    $desc=trim($_POST['description']??'');
    $always_live=empty($_POST['always_live'])?0:1;
    $event_at=$always_live?null:dt_to_mysql($_POST['event_at']??'');

    $old=$pdo->prepare("SELECT image_path,m3u8_url FROM lives WHERE id=?");
    $old->execute([$id]); $rowOld=$old->fetch() ?: [];
    $oldImg = $rowOld['image_path'] ?? null;
    $oldM3u8= $rowOld['m3u8_url'] ?? null;

    $upM3u8 = handle_m3u8_upload('m3u8_file');
    if($upM3u8){
      if($oldM3u8 && strpos($oldM3u8,'/uploads/')===0) safe_unlink($oldM3u8);
      $m3u8 = $upM3u8;
    }

    $newImg=$oldImg;
    if(!empty($_FILES['image']['name']) && is_uploaded_file($_FILES['image']['tmp_name'])){
      @mkdir($_SERVER['DOCUMENT_ROOT'].'/uploads',0775,true);
      $ext=strtolower(pathinfo($_FILES['image']['name'],PATHINFO_EXTENSION)?:'jpg');
      $ext=preg_replace('/[^a-z0-9]/','',$ext)?:'jpg';
      try{ $rand = bin2hex(random_bytes(3)); }catch(Throwable $e){ $rand = dechex(mt_rand()); }
      $fname='/uploads/live_'.time().'_'.$rand.'.'.$ext;
      if(move_uploaded_file($_FILES['image']['tmp_name'], $_SERVER['DOCUMENT_ROOT'].$fname)){
        $newImg=$fname;
        if($oldImg && $oldImg!==$newImg) safe_unlink($oldImg);
      }
    } elseif (!$oldImg) {
      // se non c'è immagine attuale e non ne carichi una, prova auto-cover
      $newImg = detect_competition_image($title, $desc) ?: null;
    }

    $stmt=$pdo->prepare("UPDATE lives
                         SET category_id=?, title=?, slug=?, m3u8_url=?, embed_code=?, image_path=?, description=?, event_at=?, always_live=?, sort_order=?, linked_live_id=?
                         WHERE id=?");
    $stmt->execute([$cat,$title,$slug,$m3u8,$embed,$newImg,$desc,$event_at,$always_live,$sort,$linked_live_id,$id]);
    header('Location: '.$self_no_edit); exit;
  }

  if(isset($_POST['delete'])){
    $id=(int)$_POST['id'];

    // recupero file da rimuovere
    $st=$pdo->prepare("SELECT image_path,m3u8_url FROM lives WHERE id=?");
    $st->execute([$id]);
    $row=$st->fetch() ?: [];

    // sicurezza: scollega eventuali live che puntano a questa
    $pdo->prepare("UPDATE lives SET linked_live_id = NULL WHERE linked_live_id = ?")->execute([$id]);

    // elimina la live (FIX: execute([ $id ]))
    $pdo->prepare("DELETE FROM lives WHERE id=?")->execute([$id]);

    // rimuovi file locali dopo la delete
    if(!empty($row['image_path'])) safe_unlink($row['image_path']);
    if(!empty($row['m3u8_url']) && strpos($row['m3u8_url'],'/uploads/')===0) safe_unlink($row['m3u8_url']);

    header('Location: '.$self_no_edit); exit;
  }
}

/* ---------- Query dati ---------- */
$playlist_token=get_setting($pdo,'playlist_token');
$cats=$pdo->query("SELECT id,name FROM categories ORDER BY name")->fetchAll();

/* Lista semplice per table join */
$allLives = $pdo->query("SELECT id,title FROM lives ORDER BY title ASC")->fetchAll(PDO::FETCH_ASSOC);

/* Edit */
$edit=null;
$editEmbeds=[];
if(isset($_GET['edit'])){
  $st=$pdo->prepare("SELECT * FROM lives WHERE id=?");
  $st->execute([(int)$_GET['edit']]);
  $edit=$st->fetch();
  if($edit){ $editEmbeds = parse_embed_variants($edit['embed_code'] ?? ''); }
}

/* Filtri tabella */
$q=trim($_GET['q']??'');
$catFilter=(int)$_GET['category_id']??0;
$activeFilter=$_GET['active']??'';

/* ======== PER-PAGE (multipli di 10) ======== */
$perPage = (int)($_GET['pp'] ?? 10);
if ($perPage < 10) $perPage = 10;
if ($perPage > 200) $perPage = 200;
if ($perPage % 10 !== 0) { // arrotonda al multiplo di 10 più vicino
  $perPage = max(10, min(200, (int)round($perPage/10)*10));
}

$where=[]; $params=[];
if($q!==''){
  $where[]="(l.title LIKE ? OR l.slug LIKE ? OR l.m3u8_url LIKE ? OR l.embed_code LIKE ? OR l.id=? OR l.description LIKE ?)";
  $params[]="%$q%"; $params[]="%$q%"; $params[]="%$q%"; $params[]="%$q%"; $params[]=ctype_digit($q)?(int)$q:-1; $params[]="%$q%";
}
if($catFilter){ $where[]="l.category_id=?"; $params[]=$catFilter; }
if($activeFilter==='1' || $activeFilter==='0'){ $where[]="l.is_active=?"; $params[]=(int)$activeFilter; }

/* ===== PAGINAZIONE ===== */
$page    = max(1, (int)($_GET['p'] ?? 1));
$cntSql  = "SELECT COUNT(*) 
            FROM lives l 
            JOIN categories c ON c.id=l.category_id".
           ($where ? " WHERE ".implode(' AND ',$where) : "");
$stc = $pdo->prepare($cntSql);
$stc->execute($params);
$total = (int)$stc->fetchColumn();

$pages  = max(1, (int)ceil($total / $perPage));
if ($page > $pages) $page = $pages;
$offset = ($page - 1) * $perPage;

/* Query con LIMIT/OFFSET */
$sql="SELECT l.*, c.name AS catname,
             l2.title AS linked_title, l2.slug AS linked_slug
      FROM lives l
      JOIN categories c ON c.id=l.category_id
      LEFT JOIN lives l2 ON l.linked_live_id = l2.id";
if($where) $sql.=" WHERE ".implode(' AND ',$where);
$sql.=" ORDER BY c.name,
               l.always_live DESC,
               CASE WHEN l.event_at IS NULL THEN 1 ELSE 0 END,
               l.event_at ASC,
               l.sort_order,
               l.id DESC
       LIMIT ? OFFSET ?";
$st=$pdo->prepare($sql);
$execParams = array_merge($params, [$perPage, $offset]);
$st->execute($execParams);
$lives=$st->fetchAll();

/* URL utili */
$scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS']!=='off') ? 'https' : 'http';
$host   = $_SERVER['HTTP_HOST'] ?? 'localhost';
$base   = $scheme.'://'.$host;
$playlist_priv = $base.'/client/playlist.php?download=0';
$playlist_pub  = $playlist_token ? ($base.'/client/playlist.php?token='.urlencode($playlist_token).'&download=0') : '';

/* Builder URL per pagine */
$qsPager = $_GET; unset($qsPager['p'], $qsPager['edit']);
function pager_url(array $qs, int $p): string {
  $qs['p'] = $p;
  return '/admin/lives.php'.'?'.http_build_query($qs);
}

require_once __DIR__.'/../partials/header.php';
?>

<?php if (isset($_GET['import_msg'])): ?>
  <div class="alert alert-info"><?= h($_GET['import_msg']) ?></div>
<?php endif; ?>

<style>
  mark{ background:#ffea7a; padding:0 .15em; border-radius:.2rem; }
  .filters-card{ background:#f8f9fb; border:1px solid #e6e9ef; border-radius:.75rem; padding:.9rem; }
  .filters-card .row > [class*="col-"]{ min-width:0; }
  @media (prefers-color-scheme: dark){
    .filters-card{ background:#171a1f; border-color:#2a2f38; }
    mark{ background:#8a6d0a; color:#fff; }
  }
  .help{ font-size:.9rem; color:#6c757d; }
  .embed-card{ background:#f8fafc; border:1px dashed #cbd5e1; border-radius:.6rem; padding:.75rem; }
  .embed-row{ border:1px solid #e6e9ef; border-radius:.5rem; padding:.5rem; background:#fff; }
  .bulk-bar{
    display:flex; gap:.5rem; align-items:center; flex-wrap:wrap;
    background:#f8f9fb; border:1px solid #e6e9ef; border-radius:.6rem; padding:.5rem .75rem; margin:.5rem 0 .75rem 0;
  }
  .bulk-count{ font-weight:700; }
  @media (prefers-color-scheme: dark){
    .bulk-bar{ background:#171a1f; border-color:#2a2f38; }
  }
  /* Picker ricerca live collegata */
  .linked-picker{ position: relative; }
  .linked-results{
    position:absolute; inset:auto 0 0 0; transform: translateY(calc(100% + 4px));
    max-height: 260px; overflow:auto; background:#fff; border:1px solid #ced4da; border-radius:.5rem; z-index:40;
    box-shadow: 0 8px 24px rgba(0,0,0,.15);
    display:none;
  }
  html[data-theme="dark"] .linked-results{ background:#111; border-color:#333; color:#eee; }
  .linked-item{ padding:.45rem .6rem; cursor:pointer; display:flex; align-items:center; gap:.5rem; }
  .linked-item:hover{ background:#f1f3f5; } html[data-theme="dark"] .linked-item:hover{ background:#1b1f25; }
  .linked-pill{ font-size:.8rem; padding:.1rem .4rem; border:1px solid #dadde2; border-radius:.35rem; }
  .linked-chosen{ margin-top:.3rem; font-size:.9rem; }
  .linked-chosen .btn{ padding:.1rem .35rem; }

  /* Pager */
  .pager{ display:flex; align-items:center; gap:.5rem; flex-wrap:wrap; }
  .pager .info{ font-size:.9rem; color:#6c757d; }
</style>

<h4 class="mb-3 d-flex flex-column flex-md-row justify-content-between align-items-md-center gap-2">
  <span>Live</span>
  <div class="d-flex flex-column flex-md-row gap-2 align-items-md-center">
    <a class="btn btn-success btn-sm" href="/admin/lives.php?export=m3u">Scarica M3U (attive)</a>
    <a class="btn btn-outline-success btn-sm" href="/admin/lives.php?export=m3u&all=1" title="Includi anche non attive">(tutte)</a>

    <form method="post" action="<?= h($self) ?>" class="d-flex gap-2">
      <button name="gen_playlist_token" class="btn btn-primary btn-sm" type="submit">Genera/Rigenera token</button>
      <?php if($playlist_token): ?>
        <button name="clear_playlist_token" class="btn btn-outline-danger btn-sm" type="submit">Revoca token</button>
      <?php endif; ?>
    </form>
  </div>
</h4>

<?php if($playlist_token): ?>
<div class="alert alert-info d-flex flex-column gap-2">
  <div><strong>Token attivo:</strong> <code><?= h($playlist_token) ?></code></div>
  <div class="d-flex flex-column flex-md-row gap-2">
    <div class="input-group input-group-sm" style="max-width:640px;">
      <span class="input-group-text">Link playlist (pubblico con token)</span>
      <input type="text" class="form-control" readonly value="<?= h($playlist_pub) ?>">
      <button class="btn btn-outline-secondary" type="button" onclick="navigator.clipboard.writeText('<?= h($playlist_pub) ?>')">Copia</button>
    </div>
    <div class="input-group input-group-sm" style="max-width:640px;">
      <span class="input-group-text">Solo attive (download)</span>
      <input type="text" class="form-control" readonly value="<?= h($base.'/client/playlist.php?token='.urlencode($playlist_token)) ?>">
      <button class="btn btn-outline-secondary" type="button" onclick="navigator.clipboard.writeText('<?= h($base.'/client/playlist.php?token='.urlencode($playlist_token)) ?>')">Copia</button>
    </div>
  </div>
</div>
<?php else: ?>
<div class="alert alert-warning">Nessun token attivo. Clicca "Genera/Rigenera token" per creare un link condivisibile.</div>
<?php endif; ?>

<!-- ===== Import CSV Lives (UI) ===== -->
<form action="lives.php?action=import_csv" method="post" enctype="multipart/form-data" class="mb-3">
  <div class="input-group">
    <input type="file" name="csv" accept=".csv,text/csv" class="form-control" required>
    <button class="btn btn-primary">Importa CSV</button>
  </div>
  <small class="text-muted">
    Colonne attese: <code>data_it, ora_it, titolo, note, is_active, always_live, slug_custom, thumb_url, fonte, label_fornitore</code>.
    La categoria verrà impostata automaticamente su <strong>Programmazione Eventi</strong>. Se <code>thumb_url</code> è vuoto,
    l’immagine viene dedotta da titolo/note usando i file in <code>/uploads/competitions/</code>.
  </small>
</form>

<?php if($edit): ?>
<!-- ======== FORM EDIT ======== -->
<form method="post" action="<?= h($self) ?>" enctype="multipart/form-data" class="row g-2 mb-3 border rounded p-3 bg-light">
  <input type="hidden" name="id" value="<?=$edit['id']?>">

  <div class="col-4">
    <label class="form-label">Titolo</label>
    <input name="title" class="form-control" value="<?=h($edit['title'])?>" required>
  </div>

  <div class="col-3">
    <label class="form-label">Categoria</label>
    <select name="category_id" class="form-select" required>
      <?php foreach($cats as $c): ?>
        <option value="<?=$c['id']?>" <?= ((int)$c['id']===(int)$edit['category_id'])?'selected':''; ?>><?=h($c['name'])?></option>
      <?php endforeach; ?>
    </select>
  </div>

  <div class="col-3">
    <label class="form-label">Data/Ora evento</label>
    <input type="datetime-local" name="event_at" id="event_at_edit" class="form-control" value="<?= h(dt_to_input($edit['event_at'] ?? null)) ?>" <?= !empty($edit['always_live']) ? 'disabled' : '' ?>>
  </div>

  <div class="col-2">
    <label class="form-label">Sempre live</label>
    <div class="form-check form-switch mt-1">
      <input class="form-check-input" type="checkbox" id="always_live_edit" name="always_live" value="1" <?= !empty($edit['always_live']) ? 'checked' : '' ?>>
      <label class="form-check-label" for="always_live_edit">Senza data</label>
    </div>
  </div>

  <div class="col-2">
    <label class="form-label">Ordine</label>
    <input name="sort_order" class="form-control" type="number" value="<?= (int)$edit['sort_order'] ?>">
  </div>

  <div class="col-5">
    <label class="form-label">URL m3u8</label>
    <input name="m3u8_url" class="form-control" value="<?=h($edit['m3u8_url'])?>" placeholder="https://.../stream.m3u8">
    <div class="help">Oppure carica un file .m3u8 (sovrascrive il campo sopra)</div>
    <input type="file" name="m3u8_file" class="form-control mt-1" accept=".m3u8,.m3u">
    <?php if(!empty($edit['m3u8_url'])): ?>
      <div class="small mt-1">Attuale: <a href="<?= h($edit['m3u8_url']) ?>" target="_blank"><?= h($edit['m3u8_url']) ?></a></div>
    <?php endif; ?>
  </div>

  <div class="col-5">
    <label class="form-label">Immagine</label>
    <input type="file" name="image" class="form-control" accept="image/*">
    <?php if($edit['image_path']): ?><div class="mt-2"><img src="<?=$edit['image_path']?>" style="height:40px"></div><?php endif; ?>
  </div>

  <!-- Collegamento ad altra live con ricerca -->
  <div class="col-6">
    <label class="form-label">Collega a un’altra live (ricerca)</label>
    <div class="linked-picker" data-picker="edit">
      <input type="hidden" name="linked_live_id" id="linked_id_edit" value="<?= (int)($edit['linked_live_id'] ?? 0) ?>">
      <input type="text" class="form-control" id="linked_search_edit" placeholder="Cerca per titolo, slug o ID…">
      <div class="linked-results" id="linked_results_edit"></div>
      <div class="linked-chosen" id="linked_chosen_edit" style="display: none;"></div>
    </div>
    <div class="help">Cerca e seleziona una live esistente da collegare. Puoi rimuovere il collegamento in qualsiasi momento.</div>
  </div>

  <!-- EMBED multipli -->
  <div class="col-12">
    <div class="d-flex justify-content-between align-items-center">
      <label class="form-label m-0">Embed (multipli)</label>
      <button type="button" class="btn btn-sm btn-primary" id="btnAddEmbedEdit">+ Aggiungi embed</button>
    </div>
    <div id="embedsEdit" class="embed-card mt-2" style="display:<?= $editEmbeds ? 'block':'none' ?>;">
      <?php if ($editEmbeds): ?><input type="hidden" name="embeds_present" value="1"><?php endif; ?>
      <?php foreach($editEmbeds as $e): ?>
        <div class="embed-row mb-2" data-row>
          <div class="row g-2">
            <div class="col-sm-4"><input name="embeds[label][]" class="form-control" placeholder="Etichetta (es. Server 1)" value="<?=h($e['label'])?>"></div>
            <div class="col-sm-2"><input name="embeds[sort][]" class="form-control" type="number" value="<?= (int)$e['sort'] ?>"></div>
            <div class="col-sm-5"><textarea name="embeds[code][]" class="form-control" rows="3" placeholder='<iframe src="https://..."></iframe>'><?=h($e['code'])?></textarea></div>
            <div class="col-sm-1 d-flex justify-content-end"><button type="button" class="btn btn-outline-danger" onclick="this.closest('[data-row]').remove()">X</button></div>
          </div>
        </div>
      <?php endforeach; ?>
    </div>
    <div class="help mt-1">Se aggiungi almeno un embed qui, verrà salvato in <code>embed_code</code> come JSON e avrà priorità.</div>
  </div>

  <!-- Legacy: embed singolo (fallback) -->
  <div class="col-12">
    <details>
      <summary class="btn btn-outline-secondary btn-sm">Embed singolo (legacy)</summary>
      <div class="mt-2">
        <label class="form-label">Embed code (iframe)</label>
        <?php $legacy_prefill = $editEmbeds ? '' : ($edit['embed_code'] ?? ''); ?>
        <textarea name="embed_code" class="form-control" rows="4" placeholder="&lt;iframe src=&quot;https://provider.example/player...&quot; allowfullscreen&gt;&lt;/iframe&gt;"><?=h($legacy_prefill)?></textarea>
      </div>
    </details>
  </div>

  <div class="col-12">
    <label class="form-label">Descrizione</label>
    <textarea name="description" class="form-control" rows="3" placeholder="Dettagli, note..."><?=h($edit['description'] ?? '')?></textarea>
  </div>

  <div class="col-12">
    <button name="update" class="btn btn-primary">Salva</button>
    <a class="btn btn-outline-secondary" href="<?= h($self_no_edit) ?>">Annulla</a>
  </div>
</form>
<script>
(() => {
  const sw=document.getElementById('always_live_edit');
  const dt=document.getElementById('event_at_edit');
  function toggle(){ if(!dt||!sw) return; dt.disabled=sw.checked; if(sw.checked) dt.value=''; }
  sw && sw.addEventListener('change',toggle); toggle();
})();
</script>
<?php else: ?>
<!-- ======== FORM CREATE ======== -->
<button class="btn btn-primary mb-2" type="button" id="toggleCreate">+ Nuova live</button>
<form id="createForm" method="post" action="<?= h($self) ?>" enctype="multipart/form-data" class="row g-2 mb-3" style="display:none">
  <div class="col-4">
    <label class="form-label">Titolo</label>
    <input name="title" class="form-control" placeholder="Es. Match XYZ" required>
  </div>

  <div class="col-3">
    <label class="form-label">Categoria</label>
    <select name="category_id" class="form-select" required>
      <option value="">Categoria</option>
      <?php foreach($cats as $c): ?><option value="<?=$c['id']?>"><?=h($c['name'])?></option><?php endforeach; ?>
    </select>
  </div>

  <div class="col-3">
    <label class="form-label">Data/Ora evento</label>
    <input type="datetime-local" name="event_at" id="event_at_new" class="form-control">
  </div>

  <div class="col-2">
    <label class="form-label">Sempre live</label>
    <div class="form-check form-switch mt-1">
      <input class="form-check-input" type="checkbox" id="always_live_new" name="always_live" value="1">
      <label class="form-check-label" for="always_live_new">Senza data</label>
    </div>
  </div>

  <div class="col-2">
    <label class="form-label">Ordine</label>
    <input name="sort_order" class="form-control" type="number" value="0" title="Ordine">
  </div>

  <div class="col-5">
    <label class="form-label">URL m3u8</label>
    <input name="m3u8_url" class="form-control" placeholder="https://.../stream.m3u8">
    <div class="help">Oppure carica un file .m3u8 (sovrascrive il campo sopra)</div>
    <input type="file" name="m3u8_file" class="form-control mt-1" accept=".m3u8,.m3u">
  </div>

  <div class="col-5">
    <label class="form-label">Immagine</label>
    <input type="file" name="image" class="form-control" accept="image/*">
  </div>

  <!-- Collegamento ad altra live con ricerca -->
  <div class="col-6">
    <label class="form-label">Collega a un’altra live (ricerca)</label>
    <div class="linked-picker" data-picker="new">
      <input type="hidden" name="linked_live_id" id="linked_id_new" value="">
      <input type="text" class="form-control" id="linked_search_new" placeholder="Cerca per titolo, slug o ID…">
      <div class="linked-results" id="linked_results_new"></div>
      <div class="linked-chosen" id="linked_chosen_new" style="display: none;"></div>
    </div>
  </div>

  <!-- EMBED multipli (crea) -->
  <div class="col-12">
    <div class="d-flex justify-content-between align-items-center">
      <label class="form-label m-0">Embed (multipli)</label>
      <button type="button" class="btn btn-sm btn-primary" id="btnAddEmbedNew">+ Aggiungi embed</button>
    </div>
    <div id="embedsNew" class="embed-card mt-2" style="display:none;"></div>
    <div class="help mt-1">Se aggiungi almeno un embed qui, verrà salvato in <code>embed_code</code> come JSON e avrà priorità.</div>
  </div>

  <!-- Legacy: embed singolo -->
  <div class="col-12">
    <details>
      <summary class="btn btn-outline-secondary btn-sm">Embed singolo (legacy)</summary>
      <div class="mt-2">
        <label class="form-label">Embed code (iframe)</label>
        <textarea name="embed_code" class="form-control" rows="4" placeholder="&lt;iframe src=&quot;https://provider.example/player...&quot; allowfullscreen&gt;&lt;/iframe&gt;"></textarea>
      </div>
    </details>
  </div>

  <div class="col-12">
    <label class="form-label">Descrizione</label>
    <textarea name="description" class="form-control" rows="3" placeholder="Dettagli, note..."></textarea>
  </div>

  <div class="col-auto">
    <button name="create" class="btn btn-primary">Aggiungi</button>
  </div>
</form>
<script>
(() => {
  const sw=document.getElementById('always_live_new');
  const dt=document.getElementById('event_at_new');
  function toggle(){ if(!dt || !sw) return; dt.disabled=sw.checked; if(sw.checked) dt.value=''; }
  sw && sw.addEventListener('change',toggle);
})();
</script>
<?php endif; ?>

<!-- ---------- FILTRI ---------- -->
<div class="filters-card mb-2">
  <form id="filters" method="get" class="row g-2 align-items-end">
    <div class="col-12 col-md-5">
      <label class="form-label">Cerca</label>
      <input type="search" class="form-control" name="q" value="<?=h($q)?>" placeholder="Titolo, slug, URL, ID o descrizione">
    </div>

    <div class="col-6 col-md-3">
      <label class="form-label">Categoria</label>
      <select name="category_id" class="form-select">
        <option value="0">Tutte</option>
        <?php foreach($cats as $c): ?>
          <option value="<?=$c['id']?>" <?= ((int)$catFilter===(int)$c['id'])?'selected':''; ?>><?=h($c['name'])?></option>
        <?php endforeach; ?>
      </select>
    </div>

    <div class="col-6 col-md-2">
      <label class="form-label">Stato</label>
      <select name="active" class="form-select">
        <option value=""  <?= $activeFilter==='' ? 'selected':''; ?>>Tutte</option>
        <option value="1" <?= $activeFilter==='1' ? 'selected':''; ?>>Attive</option>
        <option value="0" <?= $activeFilter==='0' ? 'selected':''; ?>>Non attive</option>
      </select>
    </div>

    <div class="col-6 col-md-2">
      <label class="form-label">Mostra</label>
      <select name="pp" class="form-select">
        <?php foreach([10,20,30,40,50,100,150,200] as $pp): ?>
          <option value="<?=$pp?>" <?= ($perPage===$pp)?'selected':''; ?>><?=$pp?></option>
        <?php endforeach; ?>
      </select>
    </div>

    <div class="col-12 d-flex flex-wrap gap-2 justify-content-md-end">
      <button class="btn btn-primary">Filtra</button>
      <a class="btn btn-outline-secondary" href="/admin/lives.php">Reset</a>
    </div>
  </form>
</div>

<?php
$start = $total ? ($offset+1) : 0;
$end   = min($offset + $perPage, $total);
?>
<p class="text-muted small mb-2">
  Risultati <?= $start ?>–<?= $end ?> di <?= $total ?> (pagina <?= $page ?> di <?= $pages ?> • <?= $perPage ?> per pagina)
</p>

<!-- PAGER TOP -->
<div class="pager mb-2">
  <a class="btn btn-outline-secondary btn-sm <?= $page<=1?'disabled':'' ?>" href="<?= $page<=1 ? '#' : h(pager_url($qsPager, 1)) ?>">« Prima</a>
  <a class="btn btn-outline-secondary btn-sm <?= $page<=1?'disabled':'' ?>" href="<?= $page<=1 ? '#' : h(pager_url($qsPager, $page-1)) ?>">‹ Indietro</a>
  <span class="info">Pagina <?= $page ?>/<?= $pages ?></span>
  <a class="btn btn-outline-secondary btn-sm <?= $page>=$pages?'disabled':'' ?>" href="<?= $page>=$pages ? '#' : h(pager_url($qsPager, $page+1)) ?>">Avanti ›</a>
  <a class="btn btn-outline-secondary btn-sm <?= $page>=$pages?'disabled':'' ?>" href="<?= $page>=$pages ? '#' : h(pager_url($qsPager, $pages)) ?>">Ultima »</a>
</div>

<!-- Bulk actions -->
<form id="bulkForm" method="post" action="<?= h($self) ?>" class="bulk-bar">
  <select class="form-select form-select-sm" name="bulk_action" id="bulkAction" style="max-width:220px">
    <option value="">— Azione di massa —</option>
    <option value="activate">Attiva</option>
    <option value="deactivate">Disattiva</option>
    <option value="delete">Elimina</option>
  </select>
  <button type="submit" class="btn btn-sm btn-secondary" id="bulkApply" disabled>Applica</button>
  <span class="text-muted">Selezionati: <span class="bulk-count" id="bulkCount">0</span></span>
</form>

<script>
document.getElementById('filters').addEventListener('change', function(e){
  if (e.target.matches('select')) this.submit();
});

const tgl = document.getElementById('toggleCreate');
if (tgl) tgl.addEventListener('click', ()=>{
  const f=document.getElementById('createForm');
  if(!f) return; f.style.display = (f.style.display==='none'||!f.style.display)?'block':'none';
});

window._embedRowTpl = `
  <div class="embed-row mb-2" data-row>
    <div class="row g-2">
      <div class="col-sm-4"><input name="embeds[label][]" class="form-control" placeholder="Etichetta (es. Server 1)"></div>
      <div class="col-sm-2"><input name="embeds[sort][]" class="form-control" type="number" value="0"></div>
      <div class="col-sm-5"><textarea name="embeds[code][]" class="form-control" rows="3" placeholder='<iframe src="https://..."></iframe>'></textarea></div>
      <div class="col-sm-1 d-flex justify-content-end"><button type="button" class="btn btn-outline-danger" onclick="this.closest('[data-row]').remove()">X</button></div>
    </div>
  </div>`;

function ensureEmbedsFlag(formEl){
  if (!formEl) return;
  if (!formEl.querySelector('input[name="embeds_present"]')) {
    const hid = document.createElement('input');
    hid.type = 'hidden'; hid.name = 'embeds_present'; hid.value = '1';
    formEl.appendChild(hid);
  }
}

['btnAddEmbedNew','btnAddEmbedEdit'].forEach(id=>{
  const btn = document.getElementById(id);
  if(!btn) return;
  const boxId = id==='btnAddEmbedNew' ? 'embedsNew' : 'embedsEdit';
  const box   = document.getElementById(boxId);
  btn.addEventListener('click', () => {
    if (!box) return;
    box.style.display='block';
    box.insertAdjacentHTML('beforeend', window._embedRowTpl);
    const form = btn.closest('form'); ensureEmbedsFlag(form);
  });
});

/* === Autocomplete collegamento live === */
function setupLinkedPicker(prefix, prefill){
  const input  = document.getElementById('linked_search_'+prefix);
  const hid    = document.getElementById('linked_id_'+prefix);
  const list   = document.getElementById('linked_results_'+prefix);
  const chosen = document.getElementById('linked_chosen_'+prefix);

  function renderChosen(id, title, slug, cat){
    if (!id){ chosen.style.display='none'; chosen.innerHTML=''; hid.value=''; return; }
    chosen.style.display='block';
    chosen.innerHTML = `
      <span class="me-2">Selezionato:</span>
      <span class="linked-pill">#${id}</span>
      <strong class="ms-1">${title.replace(/</g,'&lt;')}</strong>
      <span class="text-muted ms-1">(${slug ? slug.replace(/</g,'&lt;') : '-'})</span>
      <span class="text-muted ms-2">• ${cat ? cat.replace(/</g,'&lt;') : ''}</span>
      <button type="button" class="btn btn-sm btn-outline-danger ms-2" id="btnClear_${prefix}">Rimuovi collegamento</button>
    `;
    document.getElementById('btnClear_'+prefix).onclick = () => {
      hid.value = ''; input.value=''; renderChosen(0,'','',''); list.style.display='none';
    };
  }

  if (prefill && prefill.id){
    renderChosen(prefill.id, prefill.title||'', prefill.slug||'', prefill.cat||'');
    input.value = `#${prefill.id} — ${prefill.title||''}`;
  }

  let t=null;
  input?.addEventListener('input', ()=>{
    const q = input.value.trim();
    if (!q){ list.style.display='none'; return; }
    clearTimeout(t);
    t = setTimeout(async ()=>{
      try{
        const u = new URL(window.location.href);
        u.search = ''; u.hash='';
        const res = await fetch(u.pathname+'?ajax=find_live&q='+encodeURIComponent(q), {credentials:'same-origin'});
        const rows = await res.json();
        list.innerHTML = rows.map(r=>{
          const title = (r.title||'').replace(/</g,'&lt;');
          const slug  = (r.slug||'').replace(/</g,'&lt;');
          const cat   = (r.cat||'').replace(/</g,'&lt;');
          return `<div class="linked-item" data-id="${r.id}" data-title="${title}" data-slug="${slug}" data-cat="${cat}">
                    <span class="linked-pill">#${r.id}</span>
                    <span><strong>${title}</strong> <span class="text-muted">(${slug})</span></span>
                    <span class="ms-auto text-muted small">${cat}</span>
                  </div>`;
        }).join('');
        list.style.display = rows.length ? 'block' : 'none';
      }catch(e){ list.style.display='none'; }
    }, 220);
  });

  list?.addEventListener('click', (e)=>{
    const it = e.target.closest('.linked-item');
    if (!it) return;
    const id    = it.getAttribute('data-id');
    const title = it.getAttribute('data-title') || '';
    const slug  = it.getAttribute('data-slug')  || '';
    const cat   = it.getAttribute('data-cat')   || '';
    hid.value = id;
    input.value = `#${id} — ${title}`;
    list.style.display='none';
    renderChosen(id, title, slug, cat);
  });

  document.addEventListener('click', (e)=>{
    if (!e.target.closest(`[data-picker="${prefix}"]`)) list.style.display='none';
  });
}

<?php
$prefill_js = 'null';
if ($edit && !empty($edit['linked_live_id'])) {
  $stp = $pdo->prepare("SELECT l.id,l.title,l.slug,c.name AS cat FROM lives l JOIN categories c ON c.id=l.category_id WHERE l.id=?");
  $stp->execute([(int)$edit['linked_live_id']]);
  if ($r = $stp->fetch(PDO::FETCH_ASSOC)) {
    $prefill_js = json_encode($r, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
  }
}
?>
setupLinkedPicker('new', null);
setupLinkedPicker('edit', <?= $prefill_js ?>);

/* === Bulk selection UI + submit === */
const bulkForm   = document.getElementById('bulkForm');
const bulkApply  = document.getElementById('bulkApply');
const bulkCount  = document.getElementById('bulkCount');
const bulkAction = document.getElementById('bulkAction');

function selectedIds(){
  return Array.from(document.querySelectorAll('.row-check:checked'))
    .map(x => Number(x.dataset.id || x.value))
    .filter(Boolean);
}
function updateBulkUI(){
  const n = selectedIds().length;
  bulkCount.textContent = n;
  bulkApply.disabled = (n===0 || !bulkAction.value);
}
document.addEventListener('change', (e)=>{
  if (e.target && (e.target.classList.contains('row-check') || e.target.id==='checkAll' || e.target.id==='bulkAction')) updateBulkUI();
});

if (bulkForm) {
  bulkForm.addEventListener('submit', (e) => {
    const ids = selectedIds();
    if (!ids.length || !bulkAction.value) {
      e.preventDefault();
      return;
    }
    bulkForm.querySelectorAll('input[name="ids[]"]').forEach(n=>n.remove());
    ids.forEach(id => {
      const hid = document.createElement('input');
      hid.type = 'hidden';
      hid.name = 'ids[]';
      hid.value = String(id);
      bulkForm.appendChild(hid);
    });
  });
}
</script>

<!-- ======= TABLE ======= -->
<table class="table table-sm align-middle">
  <thead>
    <tr>
      <th style="width:40px">
        <input type="checkbox" id="checkAll" onclick="document.querySelectorAll('.row-check').forEach(c=>{c.checked=this.checked}); updateBulkUI();">
      </th>
      <th style="width:70px">ID</th>
      <th>Titolo</th>
      <th style="width:180px">Categoria</th>
      <th style="width:120px">Evento</th>
      <th style="width:120px">Sempre live</th>
      <th style="width:120px">Anteprima</th>
      <th style="width:100px">Collegata</th>
      <th style="width:80px">Attiva</th>
      <th style="width:80px">Ordine</th>
      <th style="width:320px"></th>
    </tr>
  </thead>
  <tbody>
<?php foreach($lives as $l): ?>
    <tr>
      <td><input type="checkbox" class="row-check" data-id="<?=$l['id']?>"></td>
      <td><?=$l['id']?></td>
      <td>
        <div><?= markQ($l['title'], $q) ?></div>
        <?php if(!empty($l['description'])): ?>
          <div class="text-muted small"><?= markQ(mb_strimwidth($l['description'],0,90,'...'), $q) ?></div>
        <?php endif; ?>
      </td>
      <td><?= markQ($l['catname'], $q) ?></td>
      <td><?= $l['event_at'] ? date('d/m/Y H:i', strtotime($l['event_at'])) : '<span class="text-muted">-</span>' ?></td>
      <td><?= !empty($l['always_live']) ? '<span class="badge bg-success">Si</span>' : 'No' ?></td>
      <td><?php if($l['image_path']): ?><img src="<?=$l['image_path']?>" style="height:40px"><?php endif; ?></td>
      <td>
        <?php if(!empty($l['linked_title'])): ?>
          <div class="small">
            <div title="Live collegata">#<?= (int)$l['linked_live_id'] ?: '' ?></div>
            <a href="/client/live.php?slug=<?= urlencode($l['linked_slug']) ?>" target="_blank">
              <?= h($l['linked_title']) ?>
            </a>
          </div>
        <?php else: ?>
          <span class="text-muted">—</span>
        <?php endif; ?>
      </td>
      <td><?= $l['is_active'] ? 'Si' : 'No' ?></td>
      <td><?=$l['sort_order']?></td>
      <td class="d-flex flex-wrap gap-2">
        <a class="btn btn-outline-secondary btn-sm"
           href="/admin/lives.php?edit=<?=$l['id']?><?= $qsNoEdit ? '&'.$qsNoEdit : '' ?>">
          Modifica
        </a>

        <?php if($l['is_active']): ?>
          <form method="post" action="<?= h($self) ?>" class="m-0">
            <input type="hidden" name="id" value="<?=$l['id']?>">
            <input type="hidden" name="active" value="0">
            <button name="set_active" class="btn btn-outline-warning btn-sm" title="Disattiva">Disattiva</button>
          </form>
        <?php else: ?>
          <form method="post" action="<?= h($self) ?>" class="m-0">
            <input type="hidden" name="id" value="<?=$l['id']?>">
            <input type="hidden" name="active" value="1">
            <button name="set_active" class="btn btn-outline-success btn-sm" title="Attiva">Attiva</button>
          </form>
        <?php endif; ?>

        <form method="post" action="<?= h($self) ?>" class="m-0" onsubmit="return confirm('Eliminare questa live?');">
          <input type="hidden" name="id" value="<?=$l['id']?>">
          <button name="delete" class="btn btn-outline-danger btn-sm">Elimina</button>
        </form>

        <a class="btn btn-outline-primary btn-sm" href="/client/live.php?slug=<?=urlencode($l['slug'])?>" target="_blank">Apri</a>
      </td>
    </tr>
<?php endforeach; ?>
  </tbody>
</table>

<!-- PAGER BOTTOM -->
<div class="pager my-2">
  <a class="btn btn-outline-secondary btn-sm <?= $page<=1?'disabled':'' ?>" href="<?= $page<=1 ? '#' : h(pager_url($qsPager, 1)) ?>">« Prima</a>
  <a class="btn btn-outline-secondary btn-sm <?= $page<=1?'disabled':'' ?>" href="<?= $page<=1 ? '#' : h(pager_url($qsPager, $page-1)) ?>">‹ Indietro</a>
  <span class="info">Pagina <?= $page ?>/<?= $pages ?></span>
  <a class="btn btn-outline-secondary btn-sm <?= $page>=$pages?'disabled':'' ?>" href="<?= $page>=$pages ? '#' : h(pager_url($qsPager, $page+1)) ?>">Avanti ›</a>
  <a class="btn btn-outline-secondary btn-sm <?= $page>=$pages?'disabled':'' ?>" href="<?= $page>=$pages ? '#' : h(pager_url($qsPager, $pages)) ?>">Ultima »</a>
</div>

<script>
typeof updateBulkUI==='function' && updateBulkUI();
</script>

<?php require_once __DIR__.'/../partials/footer.php'; ?>
