<?php
// /admin/tools/m3u8_bulk_editor.php
// Editor massivo dei link M3U8 delle lives:
// - Esporta: elenco copiabile (TSV con header: id[TAB]title[TAB]m3u8_url)
// - Importa: incolla elenco modificato, preview differenze, commit atomico
// - Parser flessibile: accetta separatori TAB, ';', ',', '|', spazi multipli
// - Accetta formati con/ senza header; con/ senza title; ignora righe invalide
// - Log delle modifiche in live_m3u8_changes; sync live_link_status.url

$page_title = 'Editor massivo M3U8';

require_once __DIR__.'/../../config.php';
require_once __DIR__.'/../../partials/admin_guard.php';

if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
ini_set('default_charset','UTF-8');
mb_internal_encoding('UTF-8');

function csrf_token() {
  if (empty($_SESSION['csrf_m3u8'])) {
    $_SESSION['csrf_m3u8'] = bin2hex(random_bytes(16));
  }
  return $_SESSION['csrf_m3u8'];
}
function csrf_check($t) {
  return isset($_SESSION['csrf_m3u8']) && hash_equals($_SESSION['csrf_m3u8'], (string)$t);
}

function fetch_lives(PDO $pdo, $onlyActive) {
  $sql = "SELECT id, title, m3u8_url FROM lives WHERE m3u8_url IS NOT NULL AND m3u8_url<>''";
  if ($onlyActive) $sql .= " AND is_active=1";
  $sql .= " ORDER BY id ASC";
  return $pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC);
}

function looks_like_header($line) {
  $l = mb_strtolower($line);
  return (strpos($l, 'id') !== false && (strpos($l,'m3u8')!==false || strpos($l,'url')!==false));
}

function smart_explode($line) {
  foreach ([ "\t", ';', ',', '|' ] as $sep) {
    if (strpos($line, $sep) !== false) return array_map('trim', explode($sep, $line));
  }
  $parts = preg_split('/\s+/', trim($line));
  return array_map('trim', $parts);
}

function parse_pasted($text) {
  $rows = [];
  $lines = preg_split("/\r\n|\n|\r/", trim($text));
  if (!$lines || (count($lines)===1 && $lines[0]==='')) return $rows;

  $start = 0;
  if (looks_like_header($lines[0])) $start = 1;

  for ($i=$start; $i<count($lines); $i++) {
    $line = trim($lines[$i]);
    if ($line === '' || strpos($line, '#') === 0) continue;

    $cols = smart_explode($line);
    if (count($cols) < 2) continue;

    // Formati accettati:
    // 1) id, url
    // 2) id, title, url
    // 3) id, ..., url (prende primo e ultimo)
    $idRaw  = $cols[0];
    $urlRaw = $cols[count($cols)-1];

    if (!ctype_digit((string)$idRaw)) continue;

    $id  = (int)$idRaw;
    $url = trim($urlRaw);

    if ($url !== '' && !preg_match('#^https?://#i', $url)) {
      if (strpos($url,'//')===0) { $url = 'https:'.$url; }
      else { continue; }
    }
    if ($url !== '' && filter_var($url, FILTER_VALIDATE_URL) === false) continue;

    $rows[$id] = $url; // l'ultima occorrenza vince
  }
  return $rows; // [ id => url ]
}

function ensure_log_table(PDO $pdo) {
  $pdo->exec("CREATE TABLE IF NOT EXISTS live_m3u8_changes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    live_id INT NOT NULL,
    old_url TEXT NOT NULL,
    new_url TEXT NOT NULL,
    changed_at DATETIME NOT NULL,
    admin_user_id INT NULL
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
}

$onlyActive = isset($_GET['only_active']) ? (int)$_GET['only_active'] : (isset($_POST['only_active'])?(int)$_POST['only_active']:1);
$mode = isset($_POST['mode']) ? $_POST['mode'] : 'export';
$message = null;
$errors = [];

if ($mode === 'commit') {
  if (!csrf_check($_POST['csrf'] ?? '')) { http_response_code(400); die('CSRF token non valido'); }
  $pasted = $_POST['pasted'] ?? '';
  $map = parse_pasted($pasted);

  if (!$map) {
    $errors[] = "Nessuna riga valida trovata nel testo incollato.";
  } else {
    $current = [];
    foreach (fetch_lives($pdo, false) as $r) {
      $current[(int)$r['id']] = (string)$r['m3u8_url'];
    }

    $todo = [];
    foreach ($map as $id => $newUrl) {
      if (!isset($current[$id])) continue;
      $old = (string)$current[$id];
      if ($newUrl !== '' && $newUrl !== $old) {
        $todo[$id] = ['old'=>$old, 'new'=>$newUrl];
      }
    }

    if (!$todo) {
      $message = "Nessuna modifica da applicare.";
    } else {
      ensure_log_table($pdo);
      $pdo->beginTransaction();
      try {
        $adminId = isset($_SESSION['user_id']) ? (int)$_SESSION['user_id'] : null;

        $upd = $pdo->prepare("UPDATE lives SET m3u8_url=? WHERE id=?");
        $ins = $pdo->prepare("INSERT INTO live_m3u8_changes (live_id, old_url, new_url, changed_at, admin_user_id) VALUES (?,?,?,?,?)");

        $count = 0;
        foreach ($todo as $id=>$pair) {
          $upd->execute([$pair['new'], $id]);
          $ins->execute([$id, $pair['old'], $pair['new'], date('Y-m-d H:i:s'), $adminId]);
          $count++;
        }

        // Sync tabella stato
        $pdo->exec("UPDATE live_link_status s
                    JOIN lives l ON l.id = s.live_id
                    SET s.url = l.m3u8_url
                    WHERE l.m3u8_url IS NOT NULL AND l.m3u8_url<>''");

        $pdo->commit();
        $message = "Modifiche applicate: {$count} live aggiornate.";
      } catch (Throwable $e) {
        $pdo->rollBack();
        $errors[] = "Errore durante il commit: ".$e->getMessage();
      }
    }
  }
}

if ($mode === 'preview') {
  if (!csrf_check($_POST['csrf'] ?? '')) { http_response_code(400); die('CSRF token non valido'); }
  $pasted = $_POST['pasted'] ?? '';
  $map = parse_pasted($pasted);
  if (!$map) {
    $errors[] = "Nessuna riga valida trovata nel testo incollato.";
  } else {
    $dbRows = fetch_lives($pdo, false);
    $byId = [];
    foreach ($dbRows as $r) $byId[(int)$r['id']] = $r;

    $preview = [];
    $changed=0; $missing=0; $same=0;
    foreach ($map as $id=>$newUrl) {
      if (!isset($byId[$id])) { $preview[] = ['id'=>$id,'title'=>'(ID non trovato)','old'=>null,'new'=>$newUrl,'status'=>'IGNORA']; $missing++; continue; }
      $old = (string)$byId[$id]['m3u8_url'];
      if ($newUrl === $old || $newUrl==='') { $preview[] = ['id'=>$id,'title'=>$byId[$id]['title'],'old'=>$old,'new'=>$newUrl,'status'=>'SAME']; $same++; }
      else { $preview[] = ['id'=>$id,'title'=>$byId[$id]['title'],'old'=>$old,'new'=>$newUrl,'status'=>'CHANGE']; $changed++; }
    }
  }
}

/* ====== Layout header ====== */
require_once __DIR__.'/../../partials/header.php';
?>
<style>
  .bulk-wrap { max-width: 1180px; margin: 16px auto; }
  .wrap { display: grid; grid-template-columns: 1fr 1fr; gap: 16px; align-items: start; }
  @media (max-width: 992px){ .wrap{ grid-template-columns: 1fr; } }
  .card { border: 1px solid var(--border-color, #e5e7eb); border-radius: 12px; padding: 16px; background: var(--body-bg, #fff); box-shadow: 0 1px 2px rgba(0,0,0,.04); }
  h1 { font-size: 1.35rem; font-weight: 800; margin: 0 0 10px; }
  h2 { margin: 0 0 8px; font-size: 1rem; font-weight: 800; }
  textarea { width: 100%; min-height: 320px; font-family: ui-monospace, SFMono-Regular, Menlo, Consolas, monospace; font-size: 13px; line-height: 1.4; padding: 8px; border-radius: 8px; border: 1px solid var(--border-color, #ccc); background: var(--body-bg, #fff); color: var(--body-text, #111); }
  .row { margin: 8px 0; }
  .btn { display: inline-block; padding: 8px 12px; border-radius: 8px; border: 1px solid rgba(0,0,0,.25); background: #f5f5f5; cursor: pointer; text-decoration: none; color: #111; }
  html[data-theme="dark"] .btn { background:#1e1f22; color:#eaeaea; border-color:#333; }
  .btn.warn { background: #f7b500; border-color: #cf9800; color: #111; }
  .btn.danger { background: #e53935; border-color: #c62828; color: #fff; }
  table { width: 100%; border-collapse: collapse; font-size: 13px; }
  th, td { text-align: left; padding: 6px 8px; border-bottom: 1px solid var(--border-color, #eee); vertical-align: top; }
  .status-CHANGE { color: #0a7; font-weight: 700; }
  .status-SAME { color: #777; }
  .status-IGNORA { color: #c00; font-weight: 700; }
  .msg { padding: 10px 12px; border-radius: 8px; margin-bottom: 12px; }
  .msg.ok { background: #e9f7ef; border: 1px solid #b7e1c4; color: #1e7e34; }
  .msg.err { background: #fdecea; border: 1px solid #f5c6cb; color: #c23616; }
  .muted { color: var(--topbar-link-muted, #6c757d); font-size: 12px; }
  .grid-2 { display:flex; gap:8px; flex-wrap:wrap; align-items:center; }
  code.k { background:#f1f3f5; padding:2px 6px; border-radius:6px; }
  .crumbs { margin: 4px 0 12px; font-size:.95rem; color: var(--topbar-link-muted, #6c757d); }
  .crumbs a { color: inherit; text-decoration: none; }
</style>

<div class="bulk-wrap container">
  <div class="crumbs"><a href="/admin/index.php">Admin</a> / Editor massivo M3U8</div>
  <h1>Editor massivo link M3U8</h1>

  <?php if ($message): ?>
    <div class="msg ok"><?= htmlspecialchars($message) ?></div>
  <?php endif; ?>
  <?php if (!empty($errors)): ?>
    <div class="msg err">
      <?= implode('<br>', array_map('htmlspecialchars', $errors)) ?>
    </div>
  <?php endif; ?>

  <div class="grid-2" style="margin-bottom:10px;">
    <form method="get" action="" class="grid-2">
      <input type="hidden" name="r" value="<?= time() ?>">
      <label><input type="checkbox" name="only_active" value="1" <?= $onlyActive? 'checked':''; ?>> solo live attive</label>
      <button class="btn" type="submit">Aggiorna elenco</button>
    </form>
    <span class="muted">Formato esportazione: <code class="k">id[TAB]title[TAB]m3u8_url</code> (con header)</span>
  </div>

  <div class="wrap">
    <div class="card">
      <h2>1) Esporta &amp; copia</h2>
      <div class="row">
        <button class="btn" type="button" id="btnSelect">Seleziona tutto</button>
        <button class="btn" type="button" id="btnCopy">Copia negli appunti</button>
        <button class="btn warn" type="button" id="btnDownload">Scarica .tsv</button>
      </div>
      <?php
        $rows = fetch_lives($pdo, $onlyActive);
        $lines = [];
        $lines[] = "id\ttitle\tm3u8_url";
        foreach ($rows as $r) {
          $t = str_replace(["\t","\r","\n"], [' ',' ',' '], $r['title']);
          $lines[] = $r['id']."\t".$t."\t".$r['m3u8_url'];
        }
        $export = implode("\n", $lines);
      ?>
      <textarea id="exportArea" readonly><?= htmlspecialchars($export) ?></textarea>
      <p class="muted">Suggerimento: copia, modifica fuori dal sito (editor/Excel), poi incolla a destra mantenendo <em>id</em> e la nuova <em>m3u8_url</em>. Il title può anche mancare.</p>
    </div>

    <div class="card">
      <h2>2) Incolla &amp; controlla</h2>
      <form method="post" action="">
        <input type="hidden" name="csrf" value="<?= htmlspecialchars(csrf_token()) ?>">
        <input type="hidden" name="only_active" value="<?= $onlyActive ? 1 : 0 ?>">
        <div class="row">
          <textarea name="pasted" id="pasted" placeholder="Incolla: id[TAB]title[TAB]m3u8_url oppure id[TAB]m3u8_url"></textarea>
        </div>
        <div class="row">
          <button class="btn" name="mode" value="preview" type="submit">Anteprima modifiche</button>
          <button class="btn danger" name="mode" value="commit" type="submit" onclick="return confirm('Confermi il commit delle modifiche?')">Applica modifiche</button>
        </div>
        <p class="muted">Parser: TAB, <code class="k">;</code>, <code class="k">,</code>, <code class="k">|</code> o spazi multipli. La prima riga d’intestazione (se presente) viene ignorata.</p>
      </form>
    </div>
  </div>

  <?php if (isset($preview)): ?>
    <div class="card" style="margin-top:16px;">
      <h2>Anteprima differenze</h2>
      <p class="muted">
        Totale righe incollate: <?= count($map) ?> —
        Cambiamenti: <b><?= $changed ?></b>,
        Uguali/empty: <b><?= $same ?></b>,
        ID non trovati: <b><?= $missing ?></b>.
      </p>
      <div style="max-height: 420px; overflow:auto; border:1px solid var(--border-color, #eee); border-radius:8px;">
        <table>
          <thead><tr><th style="width:70px;">ID</th><th>Titolo</th><th>OLD</th><th>NEW</th><th>Stato</th></tr></thead>
          <tbody>
            <?php foreach ($preview as $p): ?>
              <tr>
                <td><?= (int)$p['id'] ?></td>
                <td><?= htmlspecialchars((string)$p['title']) ?></td>
                <td><?= htmlspecialchars((string)$p['old']) ?></td>
                <td><?= htmlspecialchars((string)$p['new']) ?></td>
                <td class="status-<?= htmlspecialchars($p['status']) ?>"><?= htmlspecialchars($p['status']) ?></td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>
      <p class="muted">Solo le righe <b>CHANGE</b> verranno aggiornate al commit. Le <b>IGNORA</b> (ID non presenti) vengono scartate.</p>
    </div>
  <?php endif; ?>
</div>

<script>
const $ = sel => document.querySelector(sel);
const exportArea = $('#exportArea');
$('#btnSelect')?.addEventListener('click', ()=>{ exportArea.focus(); exportArea.select(); });
$('#btnCopy')?.addEventListener('click', async ()=>{
  exportArea.focus(); exportArea.select();
  try { await navigator.clipboard.writeText(exportArea.value); alert('Copiato!'); }
  catch(e){ document.execCommand('copy'); alert('Copiato (fallback)!'); }
});
$('#btnDownload')?.addEventListener('click', ()=>{
  const blob = new Blob([exportArea.value], {type:'text/tab-separated-values;charset=utf-8'});
  const url = URL.createObjectURL(blob);
  const a = document.createElement('a');
  a.href = url; a.download = 'lives_m3u8_export.tsv';
  document.body.appendChild(a); a.click(); a.remove();
  URL.revokeObjectURL(url);
});
</script>

<?php require_once __DIR__.'/../../partials/footer.php'; ?>
