<?php
// client/playlist.php — Max Player compat + debug
require_once __DIR__ . '/../config.php';
if (session_status() === PHP_SESSION_NONE) session_start();

ini_set('default_charset','UTF-8');
mb_internal_encoding('UTF-8');

// ---- Helpers ----
function m3u_attr(string $s): string {
  $s = html_entity_decode($s, ENT_QUOTES, 'UTF-8');
  $s = str_replace(["\r","\n"], ' ', trim($s));
  // niente virgole/pipe negli attributi
  return str_replace(['"', ',', '|'], ['\"', ' ', ' '], $s);
}
function esc_title(string $s): string {
  $s = str_replace(["\r","\n"], ' ', $s);
  // la virgola separa gli attributi dal titolo in EXTINF
  return trim(str_replace(',', ' -', $s));
}
function norm_url(string $u): string {
  $u = html_entity_decode(trim($u), ENT_QUOTES, 'UTF-8');
  $u = preg_replace('/\s+/', ' ', $u);
  return str_replace(' ', '%20', $u);
}

// ---- Parametri ----
$token = isset($_GET['token']) ? trim((string)$_GET['token']) : '';
if ($token === '' && !empty($_SERVER['PATH_INFO'])) {
  $pi = trim($_SERVER['PATH_INFO'], '/');
  $pi = preg_replace('/\.(m3u8?|txt)$/i', '', $pi);
  if ($pi !== '') $token = $pi;
}
if ($token === '') {
  http_response_code(403);
  header('Content-Type: text/plain; charset=utf-8');
  echo "Forbidden: invalid or missing token.";
  exit;
}
$download   = isset($_GET['download']) ? (trim((string)$_GET['download'])!=='0') : true;
$wantAll    = !empty($_GET['all']);     // ?all=1 include anche categorie nascoste
$useGrpLine = !empty($_GET['extgrp']);  // opzionale riga #EXTGRP
$debugMode  = !empty($_GET['debug']);   // ?debug=1 testo diagnostico
$filename   = 'playlist_'.date('Ymd_His').'.m3u';

// ---- Verifica token (utente o globale) ----
$mode = null;
try {
  $st = $pdo->prepare("SELECT id FROM users WHERE playlist_token=? LIMIT 1");
  $st->execute([$token]);
  if ($st->fetch()) $mode = 'user';
} catch (Throwable $e) {}

if ($mode === null) {
  try {
    $s = $pdo->prepare("SELECT value FROM settings WHERE `key`='playlist_token'");
    $s->execute();
    $g = trim((string)($s->fetchColumn() ?: ''));
    if ($g && hash_equals($g, $token)) $mode = 'global';
  } catch (Throwable $e) {}
}
if ($mode === null) {
  http_response_code(403);
  header('Content-Type: text/plain; charset=utf-8');
  echo "Forbidden: invalid or missing token.";
  exit;
}

// ---- Base assoluta ----
$scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS']!=='off') ? 'https' : 'http';
$host   = $_SERVER['HTTP_HOST'] ?? 'localhost';
$base   = $scheme.'://'.$host;

// ---- Query canali ----
function fetch_channels(PDO $pdo, bool $includeHidden): array {
  $where = [
    "l.is_active=1",
    "(".
      "(l.m3u8_url IS NOT NULL AND l.m3u8_url <> '')".
      " OR ".
      "(l.linked_live_id IS NOT NULL AND l2.m3u8_url IS NOT NULL AND l2.m3u8_url <> '')".
    ")"
  ];
  if (!$includeHidden) $where[] = "(c.is_hidden IS NULL OR c.is_hidden=0)";

  $sql = "SELECT
            l.id,
            l.title,
            COALESCE(NULLIF(TRIM(l.m3u8_url), ''), NULLIF(TRIM(l2.m3u8_url), '')) AS m3u8_src,
            l.image_path,
            COALESCE(NULLIF(c.name,''),'Live') AS catname
          FROM lives l
          LEFT JOIN categories c ON c.id = l.category_id
          LEFT JOIN lives l2 ON l.linked_live_id = l2.id
          WHERE ".implode(' AND ', $where)."
          ORDER BY (c.name IS NULL), c.name, l.sort_order, l.id DESC";

  return $pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC);
}

$rows = [];
$fallbackUsed = false;
try { $rows = fetch_channels($pdo, $wantAll); } catch (Throwable $e) { $rows = []; }
if (!$wantAll && count($rows) === 0) {
  try { $rows = fetch_channels($pdo, true); $fallbackUsed = true; } catch (Throwable $e) {}
}

// ---- Costruzione M3U (CRLF, UTF-8) ----
$eol = "\r\n";
$buf = "#EXTM3U".$eol;
$count = 0;

foreach ($rows as $r) {
  $url = norm_url((string)($r['m3u8_src'] ?? ''));
  if ($url === '') continue;

  // assolutizza se relativo
  $lower = strtolower(substr($url, 0, 8));
  if (strpos($lower, 'http://') !== 0 && strpos($lower, 'https://') !== 0) {
    $url = $base . ((substr($url,0,1)==='/') ? $url : '/'.$url);
  }

  $title = esc_title((string)($r['title'] ?? 'Senza titolo'));
  $group = (string)($r['catname'] ?? 'Live');

  $logo = '';
  if (!empty($r['image_path'])) {
    $logo = (stripos($r['image_path'], 'http') === 0) ? $r['image_path'] : $base.$r['image_path'];
    $logo = norm_url($logo);
  }

  $attrs = [];
  $attrs[] = 'tvg-id=""';
  $attrs[] = 'tvg-name="'.m3u_attr($title).'"';
  if ($group !== '') $attrs[] = 'group-title="'.m3u_attr($group).'"';
  if ($logo  !== '') $attrs[] = 'tvg-logo="'.m3u_attr($logo).'"';

  $buf .= "#EXTINF:-1 ".implode(' ', $attrs).",".$title.$eol;
  if ($useGrpLine && $group !== '') $buf .= "#EXTGRP:".$group.$eol;
  $buf .= $url.$eol;
  $count++;
}

// ---- DEBUG opzionale (test rapido da browser) ----
if ($debugMode) {
  header('Content-Type: text/plain; charset=utf-8');
  echo "mode: ".$mode."\n";
  echo "count: ".$count.($fallbackUsed ? " (fallback used)" : "")."\n";
  echo "sample:\n";
  $lines = explode($eol, $buf);
  echo implode("\n", array_slice($lines, 0, min(14, count($lines))));
  exit;
}

// ---- Header output per M3U (massima compatibilità) ----
while (ob_get_level()) { ob_end_clean(); }
$mime = 'audio/x-mpegurl'; // spesso più digerito di application/x-mpegURL
header('Cache-Control: no-store, no-cache, must-revalidate, private');
header('Pragma: no-cache');
header('Content-Type: '.$mime.'; charset=utf-8');
header('X-Playlist-Count: '.$count);
if ($fallbackUsed) header('X-Playlist-Fallback: used');

$disp = $download ? 'attachment' : 'inline';
header('Content-Disposition: '.$disp.'; filename="'.$filename.'"');

// calcola lunghezza dopo ogni pulizia buffer
$len = strlen($buf);
header('Content-Length: '.$len);
header('Connection: close');

if ($_SERVER['REQUEST_METHOD'] === 'HEAD') exit;
echo $buf;
