<?php
// /lib/ip_session_guard.php
// Limite IP simultanei per utente (default: 2)

if (!function_exists('ips_get_client_ip')) {
  function ips_get_client_ip(): string {
    // Se usi un reverse proxy fidato, puoi abilitare X-Forwarded-For (con cautela)
    $ip = $_SERVER['REMOTE_ADDR'] ?? '';
    $xff = trim((string)($_SERVER['HTTP_X_FORWARDED_FOR'] ?? ''));
    if ($xff) {
      // prendi il PRIMO IP della lista XFF (più vicino al client)
      $parts = array_map('trim', explode(',', $xff));
      if (!empty($parts[0])) {
        // opzionale: valida che sembri un IP “normale”
        $cand = $parts[0];
        if (filter_var($cand, FILTER_VALIDATE_IP)) $ip = $cand;
      }
    }
    return $ip ?: '0.0.0.0';
  }
}

if (!function_exists('ips_cleanup')) {
  function ips_cleanup(PDO $pdo, int $days = 30): void {
    // disattiva e/o rimuovi vecchissime sessioni inattive
    try {
      $pdo->prepare("UPDATE user_sessions SET is_active=0 WHERE last_seen < (NOW() - INTERVAL :d DAY)")
          ->execute([':d'=>$days]);
      $pdo->prepare("DELETE FROM user_sessions WHERE last_seen < (NOW() - INTERVAL :d DAY)")
          ->execute([':d'=>$days+30]);
    } catch (Throwable $e) { /* ignore */ }
  }
}

if (!function_exists('ips_count_distinct_ips')) {
  function ips_count_distinct_ips(PDO $pdo, int $userId): array {
    $st = $pdo->prepare("SELECT ip, MAX(last_seen) ls
                         FROM user_sessions
                         WHERE user_id=? AND is_active=1
                         GROUP BY ip
                         ORDER BY ls DESC");
    $st->execute([$userId]);
    $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
    return $rows; // array di ['ip'=>..., 'ls'=>...]
  }
}

if (!function_exists('ips_deactivate_by_ip')) {
  function ips_deactivate_by_ip(PDO $pdo, int $userId, string $ip): void {
    $st = $pdo->prepare("UPDATE user_sessions SET is_active=0 WHERE user_id=? AND ip=? AND is_active=1");
    $st->execute([$userId, $ip]);
  }
}

if (!function_exists('ips_register_login')) {
  /**
   * Registra la sessione al login.
   * @return bool true=ok, false=limite superato (non creato)
   */
  function ips_register_login(PDO $pdo, int $userId, int $limit = 2, bool $kickOldest = false): bool {
    if (session_status() !== PHP_SESSION_ACTIVE) session_start();
    $sid = session_id();
    $ip  = ips_get_client_ip();
    $ua  = substr((string)($_SERVER['HTTP_USER_AGENT'] ?? ''), 0, 255);

    // Sessione già registrata?
    $st = $pdo->prepare("SELECT id,is_active FROM user_sessions WHERE php_session_id=?");
    $st->execute([$sid]);
    if ($row = $st->fetch()) {
      // aggiorna IP + UA + last_seen + attiva
      $pdo->prepare("UPDATE user_sessions SET ip=?, user_agent=?, is_active=1, last_seen=NOW() WHERE id=?")
          ->execute([$ip,$ua,$row['id']]);
      return true;
    }

    // Conta IP distinti attivi
    $ips = ips_count_distinct_ips($pdo, $userId);
    $activeIps = array_column($ips, 'ip');

    if (!in_array($ip, $activeIps, true) && count($activeIps) >= $limit) {
      if ($kickOldest && $ips) {
        // disattiva l'IP meno recente
        $last = end($ips); // l'ordine è DESC -> l'ultimo è il più vecchio
        ips_deactivate_by_ip($pdo, $userId, $last['ip']);
      } else {
        return false; // limite superato
      }
    }

    // Inserisci
    $ins = $pdo->prepare("INSERT INTO user_sessions(user_id, php_session_id, ip, user_agent, created_at, last_seen, is_active)
                          VALUES(?,?,?, ?, NOW(), NOW(), 1)");
    $ins->execute([$userId, $sid, $ip, $ua]);

    return true;
  }
}

if (!function_exists('ips_touch_or_enforce')) {
  /**
   * Da chiamare su OGNI richiesta autenticata.
   * - Aggiorna last_seen della sessione corrente
   * - Se cambia IP, ri-valida il limite
   * Ritorna true=ok, false=limit hit (sessione invalidata + deve fare login)
   */
  function ips_touch_or_enforce(PDO $pdo, int $userId, int $limit = 2, bool $kickOldest = false): bool {
    if (session_status() !== PHP_SESSION_ACTIVE) session_start();
    $sid = session_id();
    $ipNow = ips_get_client_ip();
    $ua  = substr((string)($_SERVER['HTTP_USER_AGENT'] ?? ''), 0, 255);

    // Carica riga corrente
    $st = $pdo->prepare("SELECT id, ip FROM user_sessions WHERE php_session_id=? AND user_id=? AND is_active=1");
    $st->execute([$sid, $userId]);
    $row = $st->fetch(PDO::FETCH_ASSOC);

    if (!$row) {
      // Non registrata (o disattivata) -> prova a registrare (rispetta limite)
      return ips_register_login($pdo, $userId, $limit, $kickOldest);
    }

    $oldIp = $row['ip'];
    if ($oldIp !== $ipNow) {
      // Cambiato IP: verifica limite
      $ips = ips_count_distinct_ips($pdo, $userId);
      $activeIps = array_column($ips, 'ip');

      if (!in_array($ipNow, $activeIps, true) && count($activeIps) >= $limit) {
        if ($kickOldest && $ips) {
          $last = end($ips);
          ips_deactivate_by_ip($pdo, $userId, $last['ip']);
        } else {
          // invalida la sessione corrente e nega
          $pdo->prepare("UPDATE user_sessions SET is_active=0 WHERE id=?")->execute([$row['id']]);
          return false;
        }
      }

      // Aggiorna IP+UA
      $pdo->prepare("UPDATE user_sessions SET ip=?, user_agent=?, last_seen=NOW() WHERE id=?")
          ->execute([$ipNow, $ua, $row['id']]);
    } else {
      // Solo touch
      $pdo->prepare("UPDATE user_sessions SET last_seen=NOW(), user_agent=? WHERE id=?")
          ->execute([$ua, $row['id']]);
    }

    return true;
  }
}

if (!function_exists('ips_close_current')) {
  function ips_close_current(PDO $pdo): void {
    if (session_status() !== PHP_SESSION_ACTIVE) session_start();
    $sid = session_id();
    try {
      $pdo->prepare("UPDATE user_sessions SET is_active=0 WHERE php_session_id=?")->execute([$sid]);
    } catch (Throwable $e) { /* ignore */ }
  }
}
