<?php
// /lib/mailer.php
// Mailer leggero con supporto SMTP (SSL/TLS) e fallback mail()

if (!function_exists('mail_get_settings')) {
  function mail_get_settings(PDO $pdo): array {
    $keys = [
      'smtp_enabled','smtp_host','smtp_port','smtp_user','smtp_pass',
      'smtp_secure','smtp_from_email','smtp_from_name'
    ];
    $out = [];
    $st = $pdo->prepare("SELECT `key`,`value` FROM settings WHERE `key` IN (".
      implode(',', array_fill(0, count($keys), '?')).")");
    $st->execute($keys);
    foreach($st->fetchAll() as $r){ $out[$r['key']] = (string)$r['value']; }

    // default
    $out += [
      'smtp_enabled'    => '0',
      'smtp_host'       => '',
      'smtp_port'       => '587',
      'smtp_user'       => '',
      'smtp_pass'       => '',
      'smtp_secure'     => 'tls', // 'none'|'ssl'|'tls'
      'smtp_from_email' => 'no-reply@'.($_SERVER['SERVER_NAME'] ?? 'localhost'),
      'smtp_from_name'  => 'Super Sport',
    ];
    return $out;
  }

  function _mime_header_utf8(string $s): string {
    // header-safe (Subject/From Name)
    return '=?UTF-8?B?'.base64_encode($s).'?=';
  }

  function _build_message(string $fromName, string $fromEmail, string $toEmail, string $subject, string $html, ?string $text): array {
    $boundary = 'b1_'.bin2hex(random_bytes(8));
    $headers = [];
    $headers[] = 'Date: '.date('r');
    $headers[] = 'From: '._mime_header_utf8($fromName).' <'.$fromEmail.'>';
    $headers[] = 'Reply-To: '.$fromEmail;
    $headers[] = 'MIME-Version: 1.0';

    $text = $text ?? trim(preg_replace('/[ \t]+/', ' ', html_entity_decode(strip_tags($html), ENT_QUOTES, 'UTF-8')));
    $body =
      "--$boundary\r\n".
      "Content-Type: text/plain; charset=UTF-8\r\n".
      "Content-Transfer-Encoding: base64\r\n\r\n".
      chunk_split(base64_encode($text)).
      "--$boundary\r\n".
      "Content-Type: text/html; charset=UTF-8\r\n".
      "Content-Transfer-Encoding: base64\r\n\r\n".
      chunk_split(base64_encode($html)).
      "--$boundary--\r\n";

    $headers[] = 'Content-Type: multipart/alternative; boundary="'.$boundary.'"';

    // subject encoded
    $subjectEnc = _mime_header_utf8($subject);
    return [$headers, $body, $subjectEnc];
  }

  function _smtp_expect($fp, $code){
    $resp = '';
    while (($line = fgets($fp, 515)) !== false) {
      $resp .= $line;
      if (isset($line[3]) && $line[3] === ' ') break; // end of multi-line
    }
    if (substr($resp,0,3) != (string)$code) {
      throw new Exception("SMTP expected $code, got: ".$resp);
    }
    return $resp;
  }

  function _smtp_send(array $cfg, string $toEmail, string $subject, string $html, ?string $text=null): bool {
    $fromEmail = $cfg['smtp_from_email'];
    $fromName  = $cfg['smtp_from_name'];

    [$headers, $body, $subjectEnc] = _build_message($fromName, $fromEmail, $toEmail, $subject, $html, $text);

    $host = trim($cfg['smtp_host']);
    $port = (int)$cfg['smtp_port'];
    $secure = strtolower(trim($cfg['smtp_secure'] ?? 'tls')); // none|ssl|tls
    $timeout = 20;

    $transportHost = ($secure === 'ssl' ? "ssl://$host" : $host);

    $fp = @stream_socket_client("$transportHost:$port", $errno, $errstr, $timeout, STREAM_CLIENT_CONNECT);
    if (!$fp) throw new Exception("Connessione SMTP fallita: $errstr ($errno)");

    stream_set_timeout($fp, $timeout);
    _smtp_expect($fp, 220);

    $ehloName = $_SERVER['SERVER_NAME'] ?? 'localhost';
    fwrite($fp, "EHLO $ehloName\r\n");
    $resp = _smtp_expect($fp, 250);

    if ($secure === 'tls') {
      fwrite($fp, "STARTTLS\r\n");
      _smtp_expect($fp, 220);
      if (!stream_socket_enable_crypto($fp, true, STREAM_CRYPTO_METHOD_TLS_CLIENT)) {
        throw new Exception('STARTTLS fallito');
      }
      // re-EHLO
      fwrite($fp, "EHLO $ehloName\r\n");
      _smtp_expect($fp, 250);
    }

    // AUTH (se username presente)
    if (strlen(trim($cfg['smtp_user'])) > 0) {
      fwrite($fp, "AUTH LOGIN\r\n"); _smtp_expect($fp, 334);
      fwrite($fp, base64_encode($cfg['smtp_user'])."\r\n"); _smtp_expect($fp, 334);
      fwrite($fp, base64_encode($cfg['smtp_pass'])."\r\n"); _smtp_expect($fp, 235);
    }

    fwrite($fp, "MAIL FROM:<$fromEmail>\r\n"); _smtp_expect($fp, 250);
    fwrite($fp, "RCPT TO:<$toEmail>\r\n");     _smtp_expect($fp, 250);
    fwrite($fp, "DATA\r\n");                   _smtp_expect($fp, 354);

    $data =
      "Subject: $subjectEnc\r\n".
      implode("\r\n", $headers)."\r\n\r\n".
      $body.
      "\r\n.\r\n";

    fwrite($fp, $data);
    _smtp_expect($fp, 250);
    fwrite($fp, "QUIT\r\n");
    fclose($fp);
    return true;
  }

  function _mail_send_fallback(array $cfg, string $toEmail, string $subject, string $html, ?string $text=null): bool {
    $fromEmail = $cfg['smtp_from_email'];
    $fromName  = $cfg['smtp_from_name'];
    // simple HTML email
    $headers  = "MIME-Version: 1.0\r\n";
    $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
    $headers .= "From: "._mime_header_utf8($fromName)." <{$fromEmail}>\r\n";
    $headers .= "Reply-To: {$fromEmail}\r\n";
    return @mail($toEmail, _mime_header_utf8($subject), $html, $headers);
  }

  /**
   * Invia email usando configurazione in settings.
   *
   * @param string $toEmail   destinatario (singolo)
   * @param string $subject   oggetto
   * @param string $html      corpo HTML
   * @param string|null $text corpo testo (facoltativo)
   * @return bool true se ok
   */
  function send_app_mail(string $toEmail, string $subject, string $html, ?string $text=null): bool {
    global $pdo;
    if (!$pdo instanceof PDO) { throw new RuntimeException('PDO non disponibile'); }
    $cfg = mail_get_settings($pdo);
    $useSmtp = ($cfg['smtp_enabled'] === '1' && !empty($cfg['smtp_host']));
    try{
      if ($useSmtp) {
        return _smtp_send($cfg, $toEmail, $subject, $html, $text);
      }
      return _mail_send_fallback($cfg, $toEmail, $subject, $html, $text);
    } catch(Throwable $e){
      // fallback automatico a mail() se SMTP fallisce
      return _mail_send_fallback($cfg, $toEmail, $subject, $html, $text);
    }
  }
}
