<?php
// /partials/device_tracker.php
// Schema supportato: user_devices(id, user_id, device_key, ip, user_agent, created_at, last_seen)
// Obiettivo: mai sovrascrivere 2 device diversi sulla stessa riga.

function dt_client_ip(): string {
  foreach (['HTTP_CF_CONNECTING_IP','HTTP_X_FORWARDED_FOR','HTTP_X_REAL_IP','REMOTE_ADDR'] as $k) {
    if (!empty($_SERVER[$k])) {
      $ip = trim(explode(',', $_SERVER[$k])[0]);
      if (filter_var($ip, FILTER_VALIDATE_IP)) return substr($ip, 0, 45);
    }
  }
  return '';
}
function dt_user_agent(): string {
  return substr((string)($_SERVER['HTTP_USER_AGENT'] ?? ''), 0, 255);
}
function dt_cookie_key_name(): string { return 'dev_id'; }

/**
 * Genera un device_key robusto:
 * 1) se c'è il cookie dev_id valido -> usa quello
 * 2) altrimenti prova a impostarlo e usarlo
 * 3) se il browser non accetta cookie -> fallback STABILE "dk:<sha1(UA|IP|ACCEPT_LANG|PLATFORM)>"
 */
function dt_devkey_base(): string {
  if (session_status() !== PHP_SESSION_ACTIVE) @session_start();

  $cookieName = dt_cookie_key_name();
  $k = $_COOKIE[$cookieName] ?? '';

  if (preg_match('/^[A-Fa-f0-9]{40,128}$/', $k)) return $k;

  // prova a generarlo e salvarlo
  try { $k = bin2hex(random_bytes(24)); } catch (Throwable $e) { $k = bin2hex(openssl_random_pseudo_bytes(24)); }
  @setcookie($cookieName, $k, [
    'expires'=> time()+60*60*24*365*5,
    'path'=>'/',
    'secure'=> !empty($_SERVER['HTTPS']),
    'httponly'=> true,
    'samesite'=> 'Lax'
  ]);
  // se il cookie non è ancora presente in $_COOKIE in questa richiesta, imposto manualmente per l’uso corrente
  $_COOKIE[$cookieName] = $_COOKIE[$cookieName] ?? $k;

  // se il browser blocca i cookie, crea un fallback deterministico
  if (!isset($_COOKIE[$cookieName])) {
    $ua = (string)($_SERVER['HTTP_USER_AGENT'] ?? '');
    $ip = dt_client_ip();
    $al = (string)($_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '');
    $pl = (string)($_SERVER['HTTP_SEC_CH_UA_PLATFORM'] ?? '');
    $finger = $ua.'|'.$ip.'|'.$al.'|'.$pl;
    return 'dk:'.substr(sha1($finger), 0, 32);
  }

  return $k;
}

function dt_ensure_table(PDO $pdo): void {
  static $done=false; if ($done) return; $done=true;
  $pdo->exec("
    CREATE TABLE IF NOT EXISTS user_devices (
      id INT AUTO_INCREMENT PRIMARY KEY,
      user_id INT NOT NULL,
      device_key VARCHAR(128) NOT NULL,
      ip VARCHAR(45) NULL,
      user_agent VARCHAR(255) NULL,
      created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
      last_seen DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
      UNIQUE KEY uniq_user_device (user_id, device_key),
      KEY idx_user (user_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
  ");
}

/**
 * Inserisce/aggiorna il device. Se esiste già quella chiave ma UA/IP sono MOLTO diversi,
 * crea automaticamente una NUOVA chiave con suffisso per separare i device (split).
 */
function track_device(PDO $pdo, int $userId): void {
  if ($userId <= 0) return;
  dt_ensure_table($pdo);

  $baseKey = dt_devkey_base();
  $ip      = dt_client_ip();
  $ua      = dt_user_agent();

  // 1) Controlla se esiste già una riga con quella chiave
  $sel = $pdo->prepare("SELECT id, ip, user_agent FROM user_devices WHERE user_id=? AND device_key=? LIMIT 1");
  $sel->execute([$userId, $baseKey]);
  $row = $sel->fetch(PDO::FETCH_ASSOC);

  $keyToUse = $baseKey;

  if ($row) {
    $oldIp = (string)($row['ip'] ?? '');
    $oldUa = (string)($row['user_agent'] ?? '');

    // Heuristica "device diverso": UA completamente diverso o IP+UA diversi insieme
    $uaDifferent  = $oldUa !== '' && $ua !== '' && !hash_equals($oldUa, $ua);
    $ipDifferent  = $oldIp !== '' && $ip !== '' && !hash_equals($oldIp, $ip);
    $veryDifferent = $uaDifferent && $ipDifferent; // entrambi diversi -> quasi certamente altro device

    if ($veryDifferent) {
      // 2) Genera una chiave derivata e unica: base + '~' + 8 hex del fingerprint attuale
      $suffix  = substr(sha1($ua.'|'.$ip), 0, 8);
      $altKey  = $baseKey.'~'.$suffix;

      // prova a vedere se già esiste (in caso di re-visita di quel device)
      $chk = $pdo->prepare("SELECT id FROM user_devices WHERE user_id=? AND device_key=? LIMIT 1");
      $chk->execute([$userId, $altKey]);
      if ($chk->fetchColumn()) {
        $keyToUse = $altKey;
      } else {
        // crea una nuova riga per separare i device
        try {
          $ins = $pdo->prepare("INSERT INTO user_devices (user_id, device_key, ip, user_agent, created_at, last_seen)
                                VALUES (?, ?, ?, ?, NOW(), NOW())");
          $ins->execute([$userId, $altKey, $ip, $ua]);
          $keyToUse = $altKey;
        } catch (Throwable $e) {
          // se fallisce l'insert, ricadiamo sull’update della riga base
          $keyToUse = $baseKey;
        }
      }
    }
  }

  // 3) Upsert della chiave scelta (base o alternativa)
  try {
    $up = $pdo->prepare("
      INSERT INTO user_devices (user_id, device_key, ip, user_agent, created_at, last_seen)
      VALUES (?, ?, ?, ?, NOW(), NOW())
      ON DUPLICATE KEY UPDATE ip=VALUES(ip), user_agent=VALUES(user_agent), last_seen=NOW()
    ");
    $up->execute([$userId, $keyToUse, $ip, $ua]);
  } catch (Throwable $e) {
    // soft-fail: non bloccare
  }
}

/** Opzionale: check revoca device SOLO se la colonna esiste */
function dt_is_revoked(PDO $pdo, int $userId): bool {
  try {
    $db = $pdo->query("SELECT DATABASE()")->fetchColumn();
    $hasCol = $pdo->prepare("SELECT COUNT(*) FROM information_schema.COLUMNS
                             WHERE TABLE_SCHEMA=? AND TABLE_NAME='user_devices' AND COLUMN_NAME='revoked_at'");
    $hasCol->execute([$db]);
    if (!(bool)$hasCol->fetchColumn()) return false; // schema semplice: nessuna revoca

    // Se hai una colonna revoked_at e vuoi bloccare il device corrente:
    $cookie = $_COOKIE[dt_cookie_key_name()] ?? '';
    $key = $cookie && preg_match('/^[A-Fa-f0-9]{40,128}$/',$cookie) ? $cookie : dt_devkey_base();
    $q = $pdo->prepare("SELECT revoked_at FROM user_devices WHERE user_id=? AND device_key=? LIMIT 1");
    $q->execute([$userId, $key]);
    return !empty($q->fetchColumn());
  } catch (Throwable $e) {
    return false;
  }
}